import logging

from rest_framework.response import Response

from wiki.api_core.errors.rest_api_error import MoveClusterError
from wiki.api_core.framework import PageAPIView
from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.grids.move import MoveClusterSerializer, MoveMetainfoSerializer
from wiki.api_frontend.serializers.io import ok_response
from wiki.async_process.backend.consts import AsyncTaskType
from wiki.async_process.tasks.async_request_processor import delay_async_request
from wiki.pages.logic.move import move_clusters

logger = logging.getLogger(__name__)


class MoveClusterView(PageAPIView):
    """
    Переместить одиночную страницу или весь кластер

    Страница + все дети, на которых есть доступ.
    """

    serializer_class = MoveClusterSerializer
    render_blank_form_for_methods = ('PUT',)
    check_readonly_mode = True

    @raises()
    def get(self, request, *args, **kwargs):
        """
        Получить метаинформацию, необходимую для переноса страницы/табличного списка.

        %%
        $ curl -H "Authorization: OAuth <token>" "https://wiki-api.yandex-team.ru/_api/frontend/<supertag>/.move"
        %%

        ответ:
        %%(js)
        {
          "data": {
            "subpages_count": 2
          }
          "debug": {}
          "user": {}
        }
        %%

        """
        return Response(MoveMetainfoSerializer(request.page).data)

    @raises(MoveClusterError)
    def put(self, request, tag, *args, **kwargs):
        """
        Перенести весь кластер или страницу без подстраниц в другое место.
        Новый супертег страницы будет совпадать со значением в destination.

        Пример запроса:

        %%(js)
        {
          'destination': 'wiki/vodstvo'
        }
        %%

        Если в destination указана строка, которая не может быть тегом, ответ
        будет содержать сообщение об ошибке.
        """
        params = self.validate()

        old_supertag = self.request.supertag
        new_supertag = params['destination']
        clusters = {old_supertag: new_supertag}

        status, data = move_clusters(
            user=self.request.user,
            clusters=clusters,
            with_children=params['move_cluster'],
        )

        if not status == 200:
            msg = data.get('message', 'moving failed for pages')
            failed_pages = data.get('pages', [])

            if len(failed_pages) > 0:
                msg += ': %s' % (', '.join(failed_pages))

            logger.warning(msg, exc_info=True)
            raise MoveClusterError(msg)

        response = ok_response()
        response.set_cookie('just_updated', 'true')

        return response


class MoveClusterViewAsync(PageAPIView):
    serializer_class = MoveClusterSerializer
    render_blank_form_for_methods = ('PUT',)
    check_readonly_mode = True

    @raises()
    def get(self, request, *args, **kwargs):
        return Response(MoveMetainfoSerializer(request.page).data)

    @raises(MoveClusterError)
    def post(self, request, tag, *args, **kwargs):
        params = self.validate()

        old_supertag = self.request.supertag
        new_supertag = params['destination']
        clusters = {old_supertag: new_supertag}

        return delay_async_request(
            task_type=AsyncTaskType.MOVE_CLUSTER,
            task_data={
                'clusters': clusters,
                'with_children': params['move_cluster'],
                'user_id': request.user.id,
                'org_id': request.org.id if request.org else None,
            },
        )
