from django.http import Http404
from django.utils.translation import ugettext_lazy as _

from wiki.api_core.framework import WikiAPIView, raises
from wiki.api_frontend.serializers.notes import NotesSerializer
from wiki.api_frontend.serializers.pages import PageAttributesSerializer
from wiki.pages.logic import edit as edit_logic
from wiki.personalisation import quick_notes, user_cluster
from wiki.personalisation.user_cluster import NoPersonalCluster
from wiki.utils import timezone

NOTES_TAG_DATETIME_FORMAT = '%Y-%m-%dT%H:%M:%S'
NOTES_TITLE_DATETIME_FORMAT = '%Y-%m-%d %H:%M:%S'


class NotesView(WikiAPIView):
    serializer_class = NotesSerializer

    @raises()
    def post(self, request, *args, **kwargs):
        """
        Создание заметки.

        %%
        curl -H "Authorization: OAuth <token>"
        -XPOST "https://wiki-api.yandex-team.ru/_api/frontend/.notes"
        %%
        Тело запроса:
        %%(js)
        {
            "body": "**such** note"
        }
        %%

        В ответе — созданная страница в таком же формате, как она выдается
        в ручке просмотра страницы.
        """
        data = self.validate()

        now_local = timezone.now_local()
        try:
            tag = self.generate_note_tag(dt=now_local)
        except NoPersonalCluster:
            # Translators: У вас нет личного кластера.
            raise Http404(_('quick_note:No personal cluster'))

        title = self.generate_note_title(dt=now_local)
        page = edit_logic.create(
            tag=tag,
            user=request.user,
            title=title,
            body=data['body'],
        )

        return self.build_response(
            instance=page,
            serializer_class=PageAttributesSerializer,
            context={
                'user': self.request.user,
                'page': self.request.page,
            }
        )

    def generate_note_tag(self, dt):
        user = self.request.user
        personal_cluster = user_cluster.personal_cluster(user)
        notes_supertag = quick_notes.supertag_of_user_notes(user_cluster=personal_cluster)

        return '{notes}/note-{now_str}'.format(
            notes=notes_supertag,
            now_str=dt.strftime(NOTES_TAG_DATETIME_FORMAT),
        )

    @staticmethod
    def generate_note_title(dt):
        return dt.strftime(NOTES_TITLE_DATETIME_FORMAT)
