import logging

from django.shortcuts import get_object_or_404
from rest_framework.response import Response

from wiki.api_core.framework import WikiAPIView
from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.io import EmptySerializer
from wiki.pages.logic.notification import (
    disable_notification_for_user,
    enable_notification_for_user,
    is_notification_disabled_for_user,
)
from wiki.pages.models import Notification

logger = logging.getLogger(__name__)


class NotificationUserStatusView(WikiAPIView):
    """
    Управление признаком активности информационного сообщения для конкретного пользователя,
    показываемого в шапке страницы.
    """

    serializer_class = EmptySerializer

    @raises()
    def get(self, request, msg_id, **kwargs):
        """
        Получить признак активности информационного сообщения с переданным ID для текущего пользователя.

        %%
        curl -H "Authorization: OAuth <token>"
        "https://wiki-api.yandex-team.ru/_api/frontend/.notification/user_status/<msg_id>"
        %%

        Ответ
        %%(js)
        {
            "disabled": True,  // True означает, что сообщение данному пользователю показываеться никогда не будет
        }
        %%
        """

        notification = get_object_or_404(Notification, id=msg_id)

        user = self.request.user

        return Response({'disabled': is_notification_disabled_for_user(user, notification)})

    @raises()
    def post(self, request, msg_id, **kwargs):
        """
        Отключить/включить показ сообщения с переданным ID для данного пользователя.
        Запрос без доп. параметров отключает показ.
        Запрос с доп. параметром "enable" в теле запроса активирует показ сообщения,
        причем значение параметра может быть любым.

        %%
        curl -H "Authorization: OAuth <token>"
        "https://wiki-api.yandex-team.ru/_api/frontend/.notification/user_status/<msg_id>"
        %%

        Тело запроса:
        %%(js)
        {
            "enable": "any value",  // параметр  "enable" активирует показ сообщения для данного пользователя
        }
        %%
        """

        notification = get_object_or_404(Notification, id=msg_id)

        user = self.request.user
        if self.request.data.get('enable') is not None:
            enable_notification_for_user(user, notification)
        else:
            disable_notification_for_user(user, notification)

        return self.build_success_status_response()
