
from django_pgaas import atomic_retry

from wiki.api_core.errors.permissions import UserHasNoAccess
from wiki.api_core.framework import PageAPIView, raises
from wiki.api_frontend.serializers.owner import ChangeOwnerSerializer, OwnerSerializer
from wiki.pages.logic import owner as owner_logic


class OwnerView(PageAPIView):
    """
    Управление владельцем страницы.
    """

    serializer_class = ChangeOwnerSerializer

    @raises()
    def get(self, request, *args, **kwargs):
        """
        Получить владельца страницы.

        %%
        curl -H "Authorization: OAuth <token>" \
        "https://wiki-api.yandex-team.ru/_api/frontend/somepage/.owner"
        %%

        Ответ
        %%(js)
        {
            "owner": {...}  // пользователь в стандартном формате
        }
        %%
        """
        owner = request.page.owner
        return self.build_response(
            serializer_class=OwnerSerializer,
            instance={'owner': owner},
        )

    @raises()
    @atomic_retry()
    def post(self, request, *args, **kwargs):
        """
        Сменить владельца страницы

        Сменить владельца текущей страницы на сотрудника с uid=1120000000000529
        %%
        curl -H "Authorization: OAuth <token>"
        -XPOST "https://wiki-api.yandex-team.ru/_api/frontend/somepage/.owner"
        %%
        Тело запроса:
        %%(js)
        {
            "new_owner": {"uid": "1120000000000529"}
        }
        %%

        Сменить владельца текущей страницы и подстраниц, которыми он владеет
        на сотрудника с uid=1120000000000529
        %%
        curl -H "Authorization: OAuth <token>"
        -XPOST "https://wiki-api.yandex-team.ru/_api/frontend/somepage/.owner"
        %%
        Тело запроса:
        %%(js)
        {
            "new_owner": {"uid": "1120000000000529"},
            "with_subpages": true
        }
        %%

        Сменить владельца страниц в кластере с uid=1120000000009258
        на сотрудника с uid=1120000000000529
        (из интерфейса недоступна, но можно дергать через апи)
        %%
        curl -H "Authorization: OAuth <token>"
        -XPOST "https://wiki-api.yandex-team.ru/_api/frontend/somepage/.owner"
        %%
        Тело запроса:
        %%(js)
        {
            "new_owner": {"uid": "1120000000000529"},
            "old_owner": {"uid": "1120000000009258"},
            "with_subpages": true
        }
        %%
        """
        page = request.page
        user = request.user

        data = self.validate()
        with_subpages = data['with_subpages']
        new_owner = data['new_owner']
        old_owner = data.get('old_owner')

        if not old_owner:
            self.check_access_post()

            if with_subpages:
                owner_logic.change_owner_for_cluster(
                    root=page,
                    user=user,
                    new_owner=new_owner,
                    old_owner=old_owner,
                )
            else:
                owner_logic.change_owner_for_page(page=page, user=user, new_owner=new_owner)

        else:
            # при смене какого-то там оунера у каких-то там страниц внутри
            # кластера не делаем проверку доступов, но недоступные страницы
            # отфильтруются внутри функции, поэтому не проверяем и не сообщаем,
            # что к каким-то подстраницам нет доступа
            owner_logic.change_owner_for_cluster(
                root=page,
                user=user,
                new_owner=new_owner,
                old_owner=old_owner,
            )

        return self.build_success_status_response()

    def check_page_access(self):
        pass

    def check_access_post(self):
        page = self.request.page
        user = self.request.user
        if not owner_logic.can_change_owner(page, user):
            raise UserHasNoAccess('Only page owner can change page owner')
