import logging

from wiki.api_core.errors.permissions import UserHasNoAccess
from wiki.api_core.framework import PageAPIView
from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.page_readonly import PageReadonlyModeDeserializer, PageReadonlyModeSerializer
from wiki.pages.logic import readonly_mode

logger = logging.getLogger(__name__)


class PageReadonlyModeView(PageAPIView):
    """
    View для работы с режимом read-only страницы.
    """

    serializer_class = PageReadonlyModeDeserializer

    @raises()
    def get(self, request, *args, **kwargs):
        """
        Получить статус read-only режима страницы.

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" \
        -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.readonly"
        %%

        Тело ответа:

        %%(json)
        "data": {
            "is_readonly": "True",
        }
        %%
        """
        return self.build_response(instance=request.page, serializer_class=PageReadonlyModeSerializer)

    @raises()
    def post(self, request, *args, **kwargs):
        """
        Изменить режим read-only у страницы или всего кластера.

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" \
        -H -XPOST "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.readonly" \
        --data '{"is_readonly": True, "for_cluster": False}'
        %%

        В случае отсутствия ошибок тело ответа будет пустым, а статус ответа будет 200.
        """
        data = self.validate()
        self.check_access_post()
        readonly_mode.change_readonly_mode(
            request.page, request.user, is_readonly=data['is_readonly'], for_cluster=data['for_cluster']
        )
        return self.build_success_status_response()

    def check_page_access(self):
        pass

    def check_access_post(self):
        from wiki.pages import access

        if not access.is_admin(self.request.user) and self.request.user not in self.request.page.get_authors():
            raise UserHasNoAccess('Only the page author or user with admin privileges can change read-only page mode')
