import logging

from datetime import timedelta
from functools import wraps

from django import forms
from django.db import transaction
from django.http import HttpResponse, HttpResponseRedirect
from django.utils.translation import ugettext_lazy as _

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.errors.permissions import AlreadyRequestedAccess, UserHasNoAccess
from wiki.api_core.errors.rest_api_error import BillingLimitExceeded, ResourceAlreadyExists, WikiFormatterError
from wiki.api_core.framework import PageAPIView, Redirect, Response
from wiki.api_core.logic.files import _get_first_file_from_request
from wiki.api_core.not_modified import send_304_if_not_modified
from wiki.api_core.raises import raises
from wiki.api_core.serializers import PageRedirectSerializer
from wiki.api_frontend.serializers.io import ok_response
from wiki.api_frontend.serializers.pages import (
    PageAttributesSerializer,
    PageCloneSerializer,
    PageEditionSerializer,
    PageWikiTextSerializer,
)
from wiki.integrations import get_page_provider
from wiki.notifications.models import PageEvent
from wiki.pages.access import has_access, is_admin
from wiki.pages.api import get_page_section_data
from wiki.pages.constants import ReservedDeletionError
from wiki.pages.exceptions import PageTypeMismatch
from wiki.pages.logic.clone import clone_page
from wiki.pages.logic.import_file import ImportException, import_file
from wiki.pages.models import AccessRequest, CloudPage, Page, RedirectLoopException, Revision
from wiki.pages.signals import access_changed as access_changed_signal
from wiki.pages.utils.remove import delete_page
from wiki.utils import timezone
from wiki.utils.copy_on_write import is_copy_on_write
from wiki.utils.limits import limit__page_num
from wiki.utils.xaccel_header import accel_redirect_to_download

from wiki.cloudsearch.cloudsearch_client import CLOUD_SEARCH_CLIENT
from wiki.integrations.ms.wiki_to_retriever_sqs import RETRIEVER_SQS_CLIENT


logger = logging.getLogger(__name__)


def catch_redirects(method):
    """
    Декоратор для методов get/post/put/delete инстансов наследников APIView.
    Отдает корректный api ответ в случае редиректа, не пуская запрос на собственно ручку.
    """

    @wraps(method)
    def wrapper(inst, request, *args, **kwargs):
        if request.page.has_redirect() and 'noredirect' not in request.GET and 'follow_redirects' not in request.GET:
            try:
                redirect = request.page.redirect_target()
            except RedirectLoopException:
                redirect = request.page
            if redirect != request.page:
                return Response(
                    PageRedirectSerializer(
                        {
                            'redirect_to_tag': redirect.url.split('/', 1)[-1],
                            'page_type': request.page.page_type,  # именно страницы ОТКУДА редирект
                        }
                    ).data
                )

        return method(inst, request, *args, **kwargs)

    return wrapper


class PageView(PageAPIView):
    """
    View для работы со страницами и гридами.

    В зависимости от типа страницы отдает или список атрибутов страницы
    или список атрибутов табличного списка.

    Создание табличного списка реализовано в отдельной ручке.
    --Просмотр табличного списка может уехать в отдельную ручку когда-нибудь.--
    """

    serializer_class = PageEditionSerializer
    render_blank_form_for_methods = ('PUT',)
    available_content_types = ('application/json', 'application/x-www-form-urlencoded', 'multipart/form-data')
    check_readonly_mode = True

    def check_page_exists(self):
        method = self.request.method.upper()
        if method == 'POST':
            return
        if method == 'PUT':
            if self.request.page:
                raise ResourceAlreadyExists()
            return

        if hasattr(self.request, 'redirect_to'):
            raise Redirect(redirect_to=self.request.redirect_to)
        super(PageView, self).check_page_exists()

    def check_page_access(self):
        """
        Проверить, что у пользователя есть доступ на действие со страницей
        """
        http_method = self.request.method.upper()
        if http_method in {'POST', 'PUT'}:
            # we don't have page yet, so must use parent page access right
            self.check_write_access()
        elif http_method == 'DELETE':
            self.check_owner_access()
        else:
            if self.request.page.has_redirect() and 'noredirect' not in self.request.GET:
                return  # проверка прав доступа для редиректов не делается, чтобы средиректить на целевую страницу.
            try:
                super(PageView, self).check_page_access()
            except UserHasNoAccess:
                if http_method != 'GET':
                    raise

                if self.request.user.is_anonymous:
                    raise UserHasNoAccess()

                # При отказе в доступе на просмотре страницы отдаём инфо о запросе доступа
                qs = AccessRequest.objects.filter(
                    applicant_id=self.request.user.id, page_id=self.request.page.id, verdict_by=None
                )
                if qs:
                    access_request = qs[0]
                    requested_at = access_request.created_at
                    request_reason = access_request.reason
                    raise AlreadyRequestedAccess(requested_at, request_reason)
                else:
                    # пусто специально чтобы фронтэнд сматчился на пустоту.
                    raise UserHasNoAccess()

    def check_write_access(self):
        """
        Проверить, что у пользователя есть доступ на создание страницы.
        Yandex Server на ручку POST ходить не должен.
        """
        if self.request.from_yandex_server:
            raise UserHasNoAccess
        if not (
            is_admin(self.request.user)
            or has_access(
                self.request.supertag,
                self.request.user,
                privilege='write',
                current_page=self.request.page,
            )
        ):
            raise UserHasNoAccess

    def check_page_num_limit(self):
        # Если добавляется новая страница, проверяем лимит числа страниц.
        if self.request.page is None and limit__page_num.exceeded(1):
            # Translators:
            #  ru: Достигнут лимит страниц: {max_page_num} страниц.
            #  Подробнее: https://connect.yandex.ru/pricing/connect
            #  en: You have reached the maximum of pages: {max_page_num} pages.
            #  Read more: https://connect.yandex.com/pricing/connect
            raise BillingLimitExceeded(
                _('billing.limits:Page num limit {max_page_num} pages').format(max_page_num=limit__page_num.get())
            )

    def maybe_import_file(self, page):
        file_to_import = _get_first_file_from_request(self.request, raise_on_empty=False)
        if file_to_import is None:
            return None
        try:
            self.request.page = page
            return import_file(file_to_import, self.request)
        except ImportException as e:
            delete_page(page)  # Если не удалось импортировать файл, то удаляем созданную страницу
            raise InvalidDataSentError(str(e))
        except Exception:
            delete_page(page)  # Если не удалось импортировать файл, то удаляем созданную страницу
            raise

    @raises(AlreadyRequestedAccess, WikiFormatterError)
    @catch_redirects
    def get(self, request, *args, **kwargs):
        """
        Получить для просмотра список атрибутов страницы или грида.
        """
        return Response(
            PageAttributesSerializer(
                self.request.page,
                context={
                    'user': self.request.user,
                    'page': self.request.page,
                },
            ).data
        )

    @raises(UserHasNoAccess)
    def head(self, request, *args, **kwargs):
        """
        Проверить, что страница существует.

        В ответ на запрос вернется JSON, содержимое которого можно игнорировать.
        Важен только статус код 200.
        """
        # если вернуть Response, то будет content-type=application/json и пустое тело ответа – невалидный json
        return HttpResponse(status=200, content_type='text/plain')

    @raises(UserHasNoAccess, WikiFormatterError, BillingLimitExceeded)
    def post(self, request, *args, **kwargs):
        """
        Создать страницу или отредактировать существующую.

        Можно создавать подстраницы для еще не созданных страниц, например создать
        страницу "subpage" с адресом "/page/subpage", даже если страницы
        с адресом "/page" не существует.
        """
        serializer = self.get_serializer(data=request.data)
        if serializer.is_valid():
            self.check_page_num_limit()
            page = serializer.save()
            page = self.maybe_import_file(page) or page

            # Cloudsearch
            CLOUD_SEARCH_CLIENT.on_model_upsert(page)

            return Response(
                PageWikiTextSerializer(
                    page,
                    context={
                        'user': self.request.user,
                        'page': self.request.page,
                    },
                ).data
            )

        raise InvalidDataSentError(serializer.errors)

    @raises(UserHasNoAccess, ResourceAlreadyExists, WikiFormatterError, BillingLimitExceeded)
    def put(self, request, *args, **kwargs):
        """
        Создать новую страницу или ругнуться если она уже создана
        """
        # NB: проверка на существование проводится в check_page_exists,
        # которая вызывается в initial() потому что это отнаследовано от
        # PageAPIView. не самое красивое решение, но тут самое лучшее -
        # это проектировать новую апишку без supertag в урлах
        return self.post(request, *args, **kwargs)

    @raises(UserHasNoAccess)
    def delete(self, request, *args, **kwargs):
        """
        Пометить страницу как удаленную.

        На самом деле страницы из вики никогда не удаляются. Однако в API нет метода
        восстановить страницу.

        %%
        curl -H "Authorization: OAuth <token>" -X "DELETE" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/users/chapson/bropassword/"
        %%

        Ответ об успешной операции:
        %%(js)
        {
            "data": {
                "success": true          // операция прошла успешно
            },
            "user": {
                ....                     // пользователь, сделавший запрос
            },
            "debug": {
                ....             // дебаг информация, в продакшне может быть отключена
            }
        }
        %%
        """
        page = self.request.page
        # пометить страницу удаленной
        try:
            delete_page(page)

            # Cloudsearch
            CLOUD_SEARCH_CLIENT.on_model_delete(page)
            RETRIEVER_SQS_CLIENT.on_model_delete(page)

        except ReservedDeletionError:
            raise InvalidDataSentError(_('Page is protected (homepage or system page) and can\'t be deleted.'))
        except Exception as e:
            logger.exception(e)

        # создать уведомлялку
        PageEvent(
            event_type=PageEvent.EVENT_TYPES.delete,
            page=page,
            author=self.request.user,
            timeout=timezone.now() + timedelta(minutes=5),
            meta={},
        ).save()

        # сбросить кэш по сигналу
        access_changed_signal.send(sender=self.__class__, page_list=[page.supertag])

        return ok_response()


class RawPageParamsValidator(forms.Form):
    validation_error_cls = InvalidDataSentError

    follow_redirects = forms.BooleanField(
        required=False,
    )

    as_is = forms.CharField(
        required=False,
    )
    section_id = forms.IntegerField(
        min_value=1,
        required=False,
    )
    revision = forms.ModelChoiceField(
        queryset=Revision.objects.all(),
        required=False,
    )

    def clean_as_is(self):
        value = self.cleaned_data['as_is']
        if value == 'yes':
            return True
        elif value == '':
            return False
        else:
            raise forms.ValidationError("as_is should be 'yes' if given")

    def clean(self):
        cleaned_data = super(RawPageParamsValidator, self).clean()

        given_params = []
        for param in ('as_is', 'section_id', 'revision'):
            value = cleaned_data.get(param)
            if value:
                given_params.append(param)

        if len(given_params) > 1:
            params_str = ', '.join(given_params)
            msg = 'Only one parameter should be given at once, received: %s' % params_str
            for param in given_params:
                self._errors[param] = self.error_class([msg])
                del cleaned_data[param]

        return cleaned_data

    def clean_or_raise(self):
        if self.is_valid():
            return self.cleaned_data
        raise self.validation_error_cls(self.errors)


class RawPageView(PageAPIView):
    """
    View для работы wiki-разметкой страницы.
    """

    # expected_page_type = Page.TYPES.PAGE - внутри выстреливает дичью CONTENT_TYPE_IS_NOT_SUPPORTED
    # Мы уже проверяем тут во вьюшке, так как можем следовать по редиректам

    params_validator = RawPageParamsValidator

    @raises(UserHasNoAccess)
    @send_304_if_not_modified
    @catch_redirects
    @transaction.atomic()
    def get(self, request, *args, **kwargs):
        """
        Получить вики верстку всей страницы или отдельной секции или ревизии.

        Параметры:
        #|
        || **имя** | **тип** | **обязательность** | **описание** ||
        || %%section_id%% | int | не обязательный | id секции страницы,
            вики-разметку которой получаем.
            Если не указан - возвращается вики-разметка всей страницы ||
        || %%revision%% | int | не обязательный | id ревизии,
            вики-разметку которой получаем. ||
        |#

        Пример 1:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.raw"
        %%

        Ответ:

        %%(json)
        {
          "title": "Пиратские песни",
          "body": "Сто одна бутылка рома лежит в трюме, одна упала, осталось ровно сто",
          "version": "32167"
        }
        %%

        Пример 2:
        Секция это часть разметки, которая ограничена заголовком уровня n
        сверху и другим заголовком такого уровня снизу, или концом страницы,
        или экшеном include.

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.raw?section_id=2"
        %%

        Пример 3:
        Можно получить вики-разметку определенной ревизии:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.raw?revision=100500"
        %%

        В ответе будет содержаться version равный ТЕКУЩЕЙ (!) версии страницы.
        Это используется при замене текущей версии страницы какой-либо ревизией.
        Это позволяет запретить перетереть текущую версию страницы старой версией, если
        между тем как пользователь открыл на редактирование старую версию и тем как он
        нажал сохранить кто-то успел отредактировать текущую версию страницы.

        %%(json)
        {
          "title": "Пиратские песни",
          "body": "Сто одна бутылка рома лежит в трюме, одна упала, осталось ровно сто",
          "version": ""
        }
        %%

        Если передать GET-параметр "as_is" со значением "yes", то вики разметка
        не будет преобразована через preedit. Как минимум это означает, что
        в ней не будет подгружено динамическое содержимое (описания тикетов)
        и не будут учтены секции.
        Внимание! **В этом случае поменяется формат ответа!**
        Он будет не такой, как в стандартных ответах АПИ: не будет секции user, не будет
        секции debug!
        Эта возможность используется в Хемуле.

        %%(sh)
        curl -I -H "Authorization: OAuth token" \
        -H "Content-Type: application/json"
        "https://wiki-api.yandex-team.ru/_api/frontend/users/chapson/.raw?as_is=yes&format=json"
        %%
        Ответ будет выглядеть так (какие-то новые поля могут добавиться в ответ):

        %%(js)
        {
            "body": "  1. [] 123\n  1. [] 123\n  1. [] 123\n  1. [] 123\n  1. [] 123",
        }
        %%

        Вы можете использовать заголовок If-Modified-Since, чтобы сэкономить
        на времени ответа и размере ответа.
        """

        params_validator = RawPageParamsValidator(request.GET)
        params = params_validator.clean_or_raise()

        if params['as_is']:
            return accel_redirect_to_download(request.page, 'application/json')

        if not is_copy_on_write(self.request.page):
            # мы в транзакции и должны иметь консистентность между page.body и page.get_page_version()
            page = Page.objects.get(id=self.request.page.id)
        else:
            page = self.request.page

        if params['follow_redirects']:
            if page.has_redirect():
                page = page.redirect_target()

        if page.page_type not in [page.TYPES.PAGE, page.TYPES.WYSIWYG]:
            raise PageTypeMismatch(
                details={
                    'expected_type': [page.TYPES.PAGE, page.TYPES.WYSIWYG],
                    'page_type': page.page_type,
                }
            )

        body = self.get_raw_body(
            section_id=params['section_id'],
            page_like_object=params['revision'] or page,
        )
        title = page.title

        return Response(
            {
                'title': title,
                'body': body,
                # даже если была запрошена ревизия. Версия все равно актуальная.
                'version': page.get_page_version(),
                'supertag': page.supertag,
            }
        )

    def get_raw_body(self, section_id=None, page_like_object=None):
        """
        Вернуть текст вики-разметки секции страницы или страницы целиком.

        @type section_id: int
        @param section_id: id секции страницы, для которого запрашивается разметка.
            Если не указан - возвращается разметка для всей страницы
        @type page_like_object: wiki.pages.models.Revision
        @param section_id: инстанс модели ревизии, для которой запрошена разметка.
            Если не указан — возвращается разметка для последней ревизии.
        @rtype: unicode
        """
        if section_id:
            return get_page_section_data(page_body=page_like_object.body, section_number=section_id)

        return page_like_object.body


class PageCloneView(PageAPIView):
    """
    Клонирование страницы

    Content-type: application/json

    Параметры:
    #|
    || **имя** | **тип** | **обязательность** | **описание** ||
    || %%destination%% | str | !!обязательный!! | адрес новой страницы ||

    Возвращает: success: True в случае успеха

    Пример запроса:
    %%(sh)
    curl -X POST -H 'content-type: application/json' -H "Authorization: OAuth <token>" \
    "https://wiki-api.yandex-team.ru/_api/frontend/sheep/.clone" \
    --data '{"destination": "dolly"}'
    %%

    ответ:
    %%(js)
    {
        "success": True,
        "message": "",
    }
    %%
    """

    serializer_class = PageCloneSerializer
    check_readonly_mode = True

    @raises(UserHasNoAccess)
    def post(self, request, tag, *args, **kwargs):
        data = self.validate()
        page = self.request.page
        user = self.request.user
        destination_tag = data['destination']
        clone_page(page, destination_tag, [user], user)
        return self.build_success_status_response()


class BaseCloudPageView(PageAPIView):
    expected_page_type = Page.TYPES.CLOUD

    def check_page_access(self):
        """
        Проверить, что у пользователя есть доступ на действие со страницей
        """
        if self.request.page.has_redirect() and 'noredirect' not in self.request.GET:
            return  # проверка прав доступа для редиректов не делается, чтобы средиректить на целевую страницу.
        else:
            super(BaseCloudPageView, self).check_page_access()


class CloudPageView(BaseCloudPageView):
    """
    View для получения деталей облачной страницы.
    """

    @raises(UserHasNoAccess)
    @catch_redirects
    def get(self, request, *args, **kwargs):
        """
        Вернуть детали облачной страницы.

        Формат ответа:

        %%(json)
        {
            'type': 'docx|pptx|xlsx...',
            'filename': '',
            'acl_management': 'unknown|unmanaged|wiki"
            'embed': {
                'iframe_src': '',
                'edit_src': '',
            }
        }
        %%
        """
        cloud_page = CloudPage.objects.get(page_id=request.page.id)
        page_provider = get_page_provider(cloud_page.provider)
        result_data = page_provider.to_frontend_view(cloud_page)

        return Response(result_data)


class EnsureAccessView(BaseCloudPageView):
    """
    View для гарантии того что к исходному документу есть доступ у пользователя
    """

    @raises(UserHasNoAccess)
    @catch_redirects
    def post(self, request, *args, **kwargs):
        if request.user.is_anonymous:
            raise UserHasNoAccess()

        cloud_page = CloudPage.objects.get(page_id=request.page.id)
        page_provider = get_page_provider(cloud_page.provider)
        page_provider.ensure_access(cloud_page, request.user)

        return ok_response()


class CloudPageDownloadSourceView(PageAPIView):
    """
    View для загрузки исходника облачного документа.
    """

    expected_page_type = Page.TYPES.CLOUD

    @raises(UserHasNoAccess)
    def get(self, request, *args, **kwargs):
        """
        Делает редирект на url для загрузки исходника документа
        """
        cloud_page = CloudPage.objects.get(page_id=request.page.id)
        page_provider = get_page_provider(cloud_page.provider)
        download_url = page_provider.get_download_url(cloud_page)
        return HttpResponseRedirect(download_url)
