import logging

from django.contrib.auth import get_user_model
from django.http import Http404
from django.utils.translation import ugettext_lazy as _

from wiki.api_core.errors.permissions import UserHasNoAccess
from wiki.api_core.framework import PageAPIView, Response
from wiki.api_core.raises import raises
from wiki.api_core.serializers import PageRedirectSerializer
from wiki.org import org_user
from wiki.pages.models import Page
from wiki.personalisation.user_cluster import NoPersonalCluster, personal_cluster

logger = logging.getLogger(__name__)


class PersonalPageRedirectView(PageAPIView):
    """
    View для редиректа на страницы в персональном кластере (для тегов, начинающихся с ~).
    """

    def check_page_exists(self):
        # self.request.page определяется как homepage, но вообще он не должен определяться
        pass

    def check_page_access(self):
        # self.request.page определяется как homepage, но вообще он не должен определяться
        pass

    @raises()
    def get(self, request, login, tag, *args, **kwargs):
        """
        Редирект для страниц, начинающихся на ~. Не проверяется, что такая страница есть в базе.

        Вернет 404 в случае, если страница начинается на ~<login>, а логина в базе нет.

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/~thasonic/whatever"
        %%

        Ответ
        %%(json)
        {
            "redirect_to_tag": "users/thasonic/whatever",
            "page_type": "article"  # на самом деле тут могут быть и другие значения
        }
        %%

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/~"
        %%

        Ответ
        %%(json)
        {
            "redirect_to_tag": "users/thasonic",
            "page_type": "article"  # на самом деле тут могут быть и другие значения
        }

        """
        if self.request.user.is_anonymous:
            raise UserHasNoAccess()

        try:
            user = org_user().get(username=login) if login else request.user
        except get_user_model().DoesNotExist:
            raise Http404()

        try:
            redirect_to_tag = personal_cluster(user).tag
        except NoPersonalCluster:
            # Translators: У вас нет личного кластера.
            raise Http404(_('quick_note:No personal cluster'))

        if tag:
            redirect_to_tag += '/' + tag.rstrip('/')

        return Response(
            PageRedirectSerializer(
                {
                    'redirect_to_tag': redirect_to_tag,
                    'page_type': Page.TYPES.PAGE,
                }
            ).data
        )
