
from django.db import transaction
from django.utils.translation import ugettext_lazy as _

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.errors.permissions import UserHasNoAccess
from wiki.api_core.framework import PageAPIView, raises
from wiki.api_frontend.serializers.redirect import RedirectSerializer
from wiki.org import get_org
from wiki.pages import access
from wiki.pages.logic import redirect as redirect_logic
from wiki.pages.models import Page
from wiki.pages.models.page import RedirectLoopException
from wiki.utils.supertag import tag_to_supertag


class RedirectView(PageAPIView):
    """
    Управление редиректом для страницы.
    """

    serializer_class = RedirectSerializer

    def check_page_access(self):
        page = self.request.page
        user = self.request.user

        if self.request.method == 'GET':
            return

        if not access.is_admin(user) and user not in page.get_authors():
            raise UserHasNoAccess('Only the page author can set redirect')

    @raises()
    def get(self, request, *args, **kwargs):
        """
        Получить данные о том, есть ли для этой страницы редирект.

        %%
        curl -H "Authorization: OAuth <token>" \
        "https://wiki-api.yandex-team.ru/_api/frontend/somepage/.redirect"
        %%
        Тело запроса пустое.

        Тело ответа:
        %%(js)
        {
            "redirect_to_tag": "another_page" // supertag or null
        }
        %%
        """
        page = request.page
        if not page.has_redirect():
            redirect = None
        else:
            redirect = page.redirect_target()

        return self.build_response(instance={'redirect_to_tag': redirect and redirect.supertag})

    @raises()
    @transaction.atomic
    def post(self, request, *args, **kwargs):
        """
        Создать или отредактировать редирект для страницы.

        %%
        curl -H "Authorization: OAuth <token>"
        -H "Content-Type: application/json" \
        -XPOST "https://wiki-api.yandex-team.ru/_api/frontend/somepage/.redirect" \
        --data '{"redirect_to_tag": "another_page"}'
        %%

        Тело запроса:
        %%(js)
        {
            "redirect_to_tag": "another_page"
        }
        %%
        """
        page = request.page
        user = request.user

        data = self.validate()
        destination_tag = data['redirect_to_tag']
        supertag = tag_to_supertag(destination_tag)

        try:
            destination = Page.active.get(supertag=supertag, org=get_org())
        except Page.DoesNotExist:
            raise InvalidDataSentError(_('Can\'t set redirect. Destination page doesn\'t exist'))

        try:
            redirect_logic.set_redirect(page, user, destination)
        except RedirectLoopException:
            raise InvalidDataSentError(
                # Translators:
                #  ru: Не получилось создать редирект. На этой странице уже установлен редирект на текущую.
                #  en: Failed to create redirect. Target page already redirects to current page.
                _('redirects.RedirectLeadsToLoop')
            )

        return self.build_success_status_response()

    @raises()
    @transaction.atomic
    def delete(self, request, *args, **kwargs):
        """
        Удалить редирект для страницы.

        %%
        curl -H "Authorization: OAuth <token>"
        -XDELETE "https://wiki-api.yandex-team.ru/_api/frontend/somepage/.redirect"
        %%
        Тело запроса пустое.
        %%
        """
        page = request.page
        user = request.user

        redirect_logic.remove_redirect(page, user)

        return self.build_success_status_response()
