
from rest_framework.response import Response

from wiki.api_core.framework import PageAPIView
from wiki.api_core.raises import raises
from wiki.pages.logic.referrers import get_links_from_external_pages, get_links_from_wiki_pages


class ReferrersView(PageAPIView):
    @raises()
    def get(self, request, *args, **kwargs):
        """
        Возвращает список страниц, ссылающихся на данную.

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json"
        "https://wiki-api.yandex-team.ru/_api/frontend/somepage/.referrers"
        %%

        Пример ответа:

        %%(js)
        {
            "wiki_pages": [
                {
                    "url": "/otherpage",
                    "tag": "OtherPage",
                },
                ...
            ],
            "external_pages": [
                {
                    "url": "http://search.yandex-team.ru/search?text=blabla",
                    "num": 42,
                },
                ...
            ],
        }
        %%

        %%wiki_pages%% - список вики-страниц, ссылающихся на данную.
        * %%url%% - URL вики-страницы.
        * %%tag%% - тэг вики-страницы.

        %%external_pages%% - список внешних страниц, ссылающихся на данную.
        * %%url%% - URL внешней страницы (Referrer из запроса).
        * %%num%% - число переходов с этого URL на страницу.

        """

        wiki_pages = get_links_from_wiki_pages(request.page)
        wiki_pages_data = [{'url': page.url, 'tag': page.tag} for page in wiki_pages]

        external_pages = get_links_from_external_pages(request.page)
        external_pages_data = [{'url': url, 'num': num} for url, num in external_pages]

        return Response(
            {
                'wiki_pages': wiki_pages_data,
                'external_pages': external_pages_data,
            }
        )
