import logging

from django.db import OperationalError
from django.http import Http404
from django.utils.translation import ugettext_lazy as _
from django_pgaas import atomic_retry
from psycopg2 import errorcodes as pg_errorcodes

from wiki.api_core.framework import PageAPIView
from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.grids import errors
from wiki.api_frontend.serializers.grids.replace_grid import ReplaceGridSerializer
from wiki.api_frontend.serializers.io import ok_response
from wiki.cloudsearch.cloudsearch_client import CLOUD_SEARCH_CLIENT
from wiki.grids.logic.grid_replace import BadColumns, replace_grid_data
from wiki.grids.models import Grid

logger = logging.getLogger(__name__)


class ReplaceGridView(PageAPIView):
    serializer_class = ReplaceGridSerializer

    @atomic_retry()
    @raises(
        errors.MaximumRowsCountExceeded,
        errors.MaximumColumnsCountExceeded,
        errors.MaximumCellSizeExceeded,
        errors.ObjectIsLockedForUpdate,
        BadColumns,
    )
    def post(self, request, *args, **kwargs):
        try:
            grid = Grid.objects.select_for_update().get(id=request.page.id)
        except Grid.DoesNotExist:
            # Translators:
            #  ru: Табличный список не существует
            #  en: Grid does not exist
            raise Http404(_('Grid does not exist'))
        except OperationalError as err:
            if err.__cause__.pgcode == pg_errorcodes.LOCK_NOT_AVAILABLE:
                # не пятисотить, если не можем взять select_for_update у грида в течение lock_timeout и после ретраев.
                raise errors.ObjectIsLockedForUpdate()
            else:
                raise

        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        replace_grid_data(request, grid, serializer.validated_data['data'])

        # Cloudsearch
        CLOUD_SEARCH_CLIENT.on_model_upsert(grid)
        return ok_response()
