import logging

from django.conf import settings
from django.db import transaction
from rest_framework.response import Response

from wiki.api_core.errors.bad_request import AccessRequestAlreadyProcessed, InvalidDataSentError
from wiki.api_core.errors.permissions import AlreadyHasAccessError, UserHasNoAccess
from wiki.api_core.errors.rest_api_error import ResourceIsMissing
from wiki.api_core.framework import PageAPIView, WikiAPIView
from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.access_request import (
    AccessRequestSerializer,
    PendingAccessRequestAllowableSerializer,
    PendingAccessRequestDeniableSerializer,
    ProcessedAccessRequestSerializer,
)
from wiki.api_frontend.serializers.io import EmptySerializer
from wiki.api_frontend.serializers.process_access_request import ProcessAccessRequestSerializer
from wiki.api_frontend.serializers.request_access import RequestAccessSerializer
from wiki.org import org_user
from wiki.pages.access import get_access_status
from wiki.users import dao

if settings.IS_BUSINESS:
    from wiki.users.models import GROUP_TYPES

logger = logging.getLogger(__name__)


def serialize_for_apiv1(grps):
    return [{'id': getattr(grp, dao.get_id_attr_name()), 'name': grp.name} for grp in grps]


def get_applicant_groups_and_departments(applicant):
    all_groups = applicant.get_all_groups()

    if settings.IS_INTRANET:
        user_department_groups = sorted(
            (group for group in all_groups if group.is_department), key=lambda group: group.level
        )
    elif settings.IS_BUSINESS:
        user_department_groups = [group for group in all_groups if group.group_type == GROUP_TYPES.department]
    else:
        user_department_groups = []

    departments = user_department_groups

    if settings.IS_INTRANET:
        service_groups = [g for g in all_groups if g.is_service]
    elif settings.IS_BUSINESS:
        service_groups = [g for g in all_groups if g.group_type == GROUP_TYPES.group]
    else:
        service_groups = []

    services = service_groups

    return departments, services


def get_applicant_groups_and_departments_apiv1(applicant):
    departments, services = get_applicant_groups_and_departments(applicant)
    return serialize_for_apiv1(departments), serialize_for_apiv1(services)


class AccessRequestProcessingView(WikiAPIView):
    """
    Обработка запросов на доступ к странице.
    """

    serializer_class = ProcessAccessRequestSerializer

    @raises(UserHasNoAccess)
    def get(self, request, request_id, *args, **kwargs):
        """
        Получить необходимую информацию для обработки запроса доступа к странице.

        Пример запроса для запроса доступа с идентификатором запроса 43:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.requestaccess/43"
        %%

        Тело запроса пустое.

        Тело ответа зависит от статуса запроса доступа и наличия доступа.
        Для отличия между вариантами в теле запроса есть поле status.
        Возможны варианты:
          1. Запрос не обработан и доступа ещё нет, status="pending".
          2. Запрос обработан, status="processed".
          3. Запрос не обработан и доступ уже есть, status="obsolete".

        1. Тело ответа для необработанного запроса при отсутствии доступа:
        %%(js)
        {
          "status": "pending",
          "may_allow": True,  // True, если текущий пользователь имеет право менять доступ к странице,
                              // False, если не имеет (например, ответственный за группу)
          "applicant": {
             // данные про запросившего доступ пользователя в стандартном формате
          },
          "reason": "<причина запроса>",
          "requested_at": "2013-04-03T17:08:40+03:00",  // дата запроса доступа
          "applicant_services": [  // сервисные группы запросившего доступ, отсутствует при may_allow==False
            {"id": 123, "name": "group 123 name",}
            {"id": 124, "name": "group 124 name",
            ...
          ],
          "applicant_departments": [  // группы подразделений запросившего доступ, отсутствует при may_allow==False
            {"id": 125, "name": "group 125 name",}
            {"id": 126, "name": "group 126 name",}
          ]
        }
        %%

        2. Тело ответа для обработанного запроса:
        %%(js)
        {
          "status": "processed",
          "granted": False,  // False при отказе, True при выдаче
          "verdict_by": {
             // данные про обработавшего запрос пользователя в стандартном формате
          },
          "verdict_reason": "<комментарий при отказе>",  // отсутствует, если granted=True
        }
        %%

        3. Тело ответа для необработанного запроса при наличии доступа:
        %%(js)
        {
          "status": "obsolete",
        }
        %%
        """

        access_request, may_allow = ProcessAccessRequestSerializer.load(request.user, request_id)

        if access_request.verdict_by is not None:
            return Response(
                ProcessedAccessRequestSerializer(
                    {
                        'status': 'processed',
                        'granted': bool(access_request.verdict),
                        'verdict_by': access_request.verdict_by,
                        'verdict_reason': access_request.verdict_reason,
                    }
                ).data
            )

        applicant_django_user = org_user().get(username=access_request.applicant.username)
        if get_access_status(access_request.page.supertag, applicant_django_user):
            return Response(AccessRequestSerializer({'status': 'obsolete'}).data)

        data = {
            'status': 'pending',
            'may_allow': may_allow,
            'applicant': access_request.applicant,
            'reason': access_request.reason,
            'requested_at': access_request.created_at,
        }

        if may_allow:
            self.serializer_class = PendingAccessRequestAllowableSerializer

            departments, services = get_applicant_groups_and_departments_apiv1(access_request.applicant)
            data['applicant_departments'] = departments
            data['applicant_services'] = services

            return Response(PendingAccessRequestAllowableSerializer(data).data)
        else:
            return Response(PendingAccessRequestDeniableSerializer(data).data)

    @raises(
        UserHasNoAccess,
        AlreadyHasAccessError,
        AccessRequestAlreadyProcessed,
        ResourceIsMissing,
    )
    @transaction.atomic
    def post(self, request, request_id, *args, **kwargs):
        """
        Обработать запрос доступа к странице.

        Пример запроса для запроса доступа с идентификатором запроса 43:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -X "POST" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.requestaccess/43" \
        --data 'тело запроса'
        %%

        Тело запроса:
        %%(json)
        {
          "action": "deny",  // "deny" для отказа,
                             // "allow_applicant" для разрешения только пользователю
                             // "allow_groups" для разрешения группам,
          "verdict_reason": "<комментарий при отказе>",  // используется только при action="deny"
          "groups": [11, 25, ],  // используется только при action="allow_groups"
          "departments": [16, 35, ],  // используется только при action="allow_groups"
        }
        %%

        В groups указываются и все выбранные группы пользователя, и выбранные группы, к которым
        пользователь не принадлежит.

        Если нет ошибок, ответ будет пустой.
        """
        serializer = self.get_serializer(data=request.data)

        if not serializer.is_valid():
            raise InvalidDataSentError(serializer.errors)

        serializer.save(request.user, request_id)
        return Response(EmptySerializer().data)


class RequestAccessView(PageAPIView):
    """
    Запрос на доступ к странице.
    """

    serializer_class = RequestAccessSerializer
    render_blank_form_for_methods = ('PUT',)

    def check_page_access(self):
        pass

    @transaction.atomic
    @raises(AlreadyHasAccessError)
    def put(self, request, *args, **kwargs):
        """
        Запросить доступ к странице.

        Пример запроса доступа к странице "/wat":
        %%
        PUT /_api/frontend/wat/.requestaccess
        %%
        Тело запроса:
        %%(js)
        {
          "reason": "<причина запроса>"
        }
        %%

        Если нет ошибок, ответ будет пустой.
        """
        serializer = self.get_serializer(data=request.data)

        if not serializer.is_valid():
            raise InvalidDataSentError(serializer.errors)

        serializer.save(request.user, request.page)
        return Response(EmptySerializer().data)
