from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.framework import PageAPIView
from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.request_ownership import RequestOwnershipSerializer


class RequestOwnershipView(PageAPIView):
    serializer_class = RequestOwnershipSerializer
    render_blank_form_for_methods = ('PUT',)

    @raises(InvalidDataSentError)
    def put(self, request, *args, **kwargs):
        """
        Запросить владение кластером страниц.

        Доступ запрашивается только для страниц,
        которые принадлежат владельцу корневой страницы кластера
        (в кластере могут быть страницы с другими владельцами).

        Если владелец корневой страницы кластера – бывший сотрудник,
        то уведомление отправляется на tools@.

        Иначе уведомление отправляется владельцу корневой страницы кластера.

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        -XPUT "https://wiki-api.yandex-team.ru/_api/frontend/somepage/.request_ownership"
        %%

        Тело запроса:

        %%(js)
        {
          "root_page_only": <bool>,
          "reason": <str>,
        }
        %%

        %%root_page_only%% (необязательный параметр, по умолчанию False) - если True,
        то запросить владение только корневой страницей кластера,
        иначе запросить владение всеми страницами кластера.

        %%reason%% (обязательный параметр) - причина, по которой хочется завладеть кластером.

        Тело ответа, если нет ошибок:
        %%(js)
        {
            "success": True,
            "message": "",
        }
        %%
        """
        serializer = self.get_serializer(data=request.data)

        if not serializer.is_valid():
            raise InvalidDataSentError(serializer.errors)

        serializer.save(request.user, request.page)

        return self.build_success_status_response()
