from django.contrib.auth import get_user_model
from django.http import Http404
from rest_framework.response import Response

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.errors.permissions import UserHasNoAccess
from wiki.api_core.framework import PageAPIView
from wiki.api_core.raises import raises
from wiki.api_core.serializers import UserSerializer
from wiki.api_frontend.serializers.request_page_author import RequestPageAuthorSerializer
from wiki.org import org_user
from wiki.pages import access
from wiki.pages.logic import authors as authors_logic


class RequestPageAuthorView(PageAPIView):
    """
    Запрос на добавление в список авторов страницы.
    """

    serializer_class = RequestPageAuthorSerializer
    render_blank_form_for_methods = ('PUT', 'POST')

    @raises(InvalidDataSentError)
    def put(self, request, *args, **kwargs):
        """
        Запросить добавление в список авторов страницы.

        Запрос может сделать только пользователь, у которого есть доступ к странице.

        Если автор запроса - владелец верхнеуровнего раздела,
        то ему выдается авторство без дополнительных подтверждений,
        иначе - авторам страницы отправляется уведомление с экшен ссылкой - добавить этого пользователя
        в список авторов или отклонить.

        Если все авторы страницы – бывшие сотрудники, то уведомление отправляется на tools@.

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        -XPUT "https://wiki-api.yandex-team.ru/_api/frontend/somepage/.request_page_author"
        %%

        Тело запроса:

        %%(js)
        {
          "reason": <str>,
        }
        %%

        %%reason%% (обязательный параметр) - причина, по которой хочется стать автором страницы.

        Тело ответа, если нет ошибок и запрос сохранен:
        %%(js)
        {
            "success": True,
            "message": "The request was saved",
        }
        %%

        Тело ответа, если новый автор был сразу добавлен в число авторов, без дополнительных подтверждений:
        %%(js)
        {
            "success": True,
            "new_author": {
                // Данные про нового автора страницы в стандартном формате
            },
        }
        %%

        Тело ответа, если пользователь уже есть среди авторов:
        %%(js)
        {
            "success": True,
            "message": "The user is already in the authors list",
        }
        %%
        """
        user = self.request.user
        page = self.request.page
        serializer = self.get_serializer(data=request.data)

        if not serializer.is_valid():
            raise InvalidDataSentError(serializer.errors)

        parent_authors = set()
        for parent_page in page.ancestors:
            if parent_page.supertag != 'users':
                parent_authors.update(parent_page.get_authors())

        if user in page.get_authors():
            return self.build_success_status_response(message='The user is already in the authors list')

        if access.is_admin(user) or user in parent_authors or page.is_all_authors_dismissed():
            authors_logic.add_user_to_authors(page, user, user)
            return Response({'new_author': UserSerializer(user).data})

        serializer.save(user, page)

        return self.build_success_status_response(message='The request was saved')

    @raises(UserHasNoAccess)
    def post(self, request, *args, **kwargs):
        """
        Добавить пользователя с переданным логином в список авторов текущей страницы.

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        -XPOST "https://wiki-api.yandex-team.ru/_api/frontend/somepage/.request_page_author"
        %%

        Тело запроса:

        %%(js)
        {
          "new_author": <login>,
        }
        %%

        Тело ответа, если нет ошибок:
        %%(js)
        {
            "success": True,
            "new_author": {
                // Данные про нового автора страницы в стандартном формате
            },
        }
        %%
        """
        login = self.request.data.get('new_author')

        try:
            new_author = org_user().get(username=login)
        except get_user_model().DoesNotExist:
            raise Http404('Can\'t find user with login=%s' % login)

        authors_logic.add_user_to_authors(self.request.page, self.request.user, new_author)

        return Response({'new_author': UserSerializer(new_author).data})

    def check_access_post(self):
        page = self.request.page
        user = self.request.user
        if not authors_logic.can_change_authors(page, user):
            raise UserHasNoAccess('Only authors of page can change list of page authors')
