from rest_framework.response import Response
from rest_framework.serializers import Serializer
from typing import Dict
from wiki.api_core.framework import WikiAPIView
from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.user_identity import UserIdentityListField
from wiki.users.dao import get_users_by_identity


class IdentitySerializer(Serializer):  # noqa
    identities = UserIdentityListField()


def serialize_user(user) -> Dict[str, str]:
    return {
        'login': user.get_username(),
        'name': user.staff.get_full_name(),
        'cloud_uid': user.get_cloud_uid(),
        'uid': user.get_uid(),
    }


class ResolveUsersByIdentities(WikiAPIView):
    @raises()
    def post(self, request):
        serializer = IdentitySerializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        identities = serializer.validated_data['identities']
        users = get_users_by_identity(identities, panic_if_missing=True, prefetch_staff=True)

        data = {'users': [serialize_user(user) for user in users]}
        return Response(data=data, status=200)
