from django.db.models import Q, QuerySet

from rest_framework.response import Response
from rest_framework.serializers import Serializer, ListField, CharField

from wiki.api_core.framework import WikiAPIView
from wiki.api_core.raises import raises
from wiki.org import org_user
from wiki.sync.connect.base_organization import BaseOrganization

from typing import Dict, List, Tuple


class LoginListField(ListField):
    def __init__(self, **kwargs):
        kwargs['child'] = CharField()
        super().__init__(**kwargs)


class LoginSerializer(Serializer):  # noqa
    logins = LoginListField()


def divide_by_logins_and_emails(logins_and_email: List[str]) -> Tuple[List[str], List[str]]:
    logins, emails = [], []
    for elem in logins_and_email:
        elem = elem.replace(':', '@')
        if '@' in elem:
            emails.append(elem)
        else:
            logins.append(elem)
    return logins, emails


def get_users_by_logins_and_emails(
    logins: List[str], emails: List[str], prefetch_staff=False, organization: BaseOrganization = None
) -> QuerySet:
    if organization:
        mdl = organization.get_users()
    else:
        mdl = org_user()

    if prefetch_staff:
        mdl = mdl.select_related('staff')

    if not logins and not emails:
        return mdl.none()

    query = Q(username__in=logins + emails)
    if emails:
        query |= Q(email__in=emails)
    for login in logins:
        query |= Q(username__startswith=login + '@')

    return mdl.filter(query)


def serialize_user(user) -> Dict[str, str]:
    return {
        'login': user.get_username(),
        'name': user.staff.get_full_name(),
        'cloud_uid': user.get_cloud_uid(),
        'uid': user.get_uid(),
    }


class ResolveUsersByLogins(WikiAPIView):
    @raises()
    def post(self, request):
        """
        Получить информацию о пользователях по "login | login@domain | login|domain"

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -X "POST" -H "Content-Type: application/json" \
        "https://wiki-api.yandex.ru/_api/frontend/.resolve_users_logins" \
        --data 'тело запроса'
        %%

        Тело запроса:

        %%(js)
        {
            "logins": [
                "login",
                "login@domain",
                "login:domain"
            ]
        }
        %%

        Ответ:

        %%(js)
        {
            "users": [
                {
                    "login": "username",
                    "name": "full_name",
                    "cloud_uid": cloud_uid,
                    "uid": uid,
                },
            ]
        }
        %%
        """

        serializer = LoginSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        logins_and_emails = serializer.validated_data['logins']
        logins, emails = divide_by_logins_and_emails(logins_and_emails)
        users = get_users_by_logins_and_emails(logins, emails, prefetch_staff=True)

        data = {'users': [serialize_user(user) for user in users]}
        return Response(data=data, status=200)
