import logging

from django.shortcuts import get_object_or_404
from rest_framework.response import Response

from wiki.api_core.framework import PageAPIView
from wiki.api_core.raises import raises
from wiki.api_core.utils import paginated
from wiki.api_frontend.serializers.revisions import (
    PageEventsArraySerializer,
    RevisionsArraySerializer,
    RevisionSerializer,
)
from wiki.utils.diff import DifflibDiff

logger = logging.getLogger(__name__)

_diff = DifflibDiff(return_type='chunks')


class RevisionListView(PageAPIView):
    """
    Ревизии страниц и гридов
    """

    serializer_class = RevisionsArraySerializer

    @raises()
    @paginated
    def get(self, request, start=None, limit=None, *args, **kwargs):
        """
        Вернуть список ревизий для страницы.

        Используйте параметры start и limit для получения новых страниц:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/username/.revisions?start=10&limit=100"
        %%

        %%(json)
        {
            total: 1,
            data: [
                {
                    id: 9078564,
                    created_at: "2014-06-18T17:40:40",
                    author: {
                        // стандартное представление автора
                    }
                }
            ]}
        %%
        """
        return Response(self.serializer_class(request.page, start=start, limit=limit).data)


class PageEventsView(PageAPIView):
    """
    События страницы
    """

    @raises()
    @paginated
    def get(self, request, start=None, limit=None, *args, **kwargs):
        """
        Вернуть список событий для страницы.

        Параметры:
        #|
        || **имя** | **тип** | **обязательность** | **описание** ||
        || %%start%% | int |необязательный | первый элемент в текущей странице паджинации ||
        || %%limit%% | int |необязательный | максимальное количество возвращаемых результатов ||
        || %%event_types%% | str or int | не обязательный
        | список типов событий для ограничения выборки. По умолчанию возвращаются события всех типов.
        В списке могут быть перечислены либо цифровые идентификаторы типов событий, либо их текстовые наименования ||
        |#

        Список наименований всех имеющихся типов событий с их идентификаторами:
        %%(code)
        1: 'create',
        2: 'delete',
        3: 'edit',
        4: 'access',
        5: 'request_access',
        6: 'keywords',
        7: 'rename',
        8: 'add_file',
        9: 'delete_file',
        10: 'add_comment',
        11: 'delete_comment',
        12: 'watch',
        13: 'unwatch',
        14: 'resolve_access',
        15: 'closed_page_access',
        16: 'change_owner',
        17: 'move_cluster',
        18: 'remove_comment',
        19: 'request_ownership',
        20: 'change_to_redirect',
        21: 'revert_from_redirect',
        22: 'mark_obsolete',
        23: 'mark_actual',
        24: 'subscribe_other_user',
        25: 'change_authors',
        26: 'request_page_author',
        27: 'add_author',
        28: 'change_access',
        29: 'enable_page_readonly',
        30: 'disable_page_readonly',

        %%

        Вернуть события всех типов
        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/username/somepage/.events"
        %%

        Вернуть события только определенных типов
        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/username/somepage/.events?event_types=1,3,5"
        %%

        Вернуть события только определенных типов
        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/username/somepage/.events?event_types=create,edit"
        %%

        Пример ответа:
        %%(js)
        {
            "debug": { },
            "data": {
                "total": 7,
                "data": [
                    {
                        "additional_fields": {},
                        "created_at": "2013-04-03T17:08:40+03:00",
                        "author": {
                            "uid": 1120000000008011,
                            "login": "elisei",
                            "display": "Алексей Микерин",
                            "email": "elisei@yandex-team.ru",
                            "first_name": "Алексей",
                            "last_name": "Микерин",
                            "department": "Группа разработки Вики",
                            "is_dismissed": false,
                            "is_admin": false
                        },
                        "event_type": "edit",
                        "revision": {
                            "id": 5883576,
                            "created_at": "2013-04-03T17:08:40+03:00",
                            "author": {
                                "uid": 1120000000008011,
                                "login": "elisei",
                                "display": "Алексей Микерин",
                                "email": "elisei@yandex-team.ru",
                                "first_name": "Алексей",
                                "last_name": "Микерин",
                                "department": "Группа разработки Вики",
                                "is_dismissed": false,
                                "is_admin": false
                            }
                        }
                    },
                    ....
                ]
            },
            "user": {},
        %%
        """
        event_types = self.request.GET.get('event_types', None)

        return Response(
            PageEventsArraySerializer(self.request.page, start=start, limit=limit, event_types=event_types).data
        )


class RevisionView(PageAPIView):
    """
    Ревизия страницы.
    """

    @raises()
    @paginated
    def get(self, request, revision_id, *args, **kwargs):
        """
        Вернуть данные ревизии для страницы.

        Пример запроса, возвращающего ревизию 43 в bem json.

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.revisions/43"
        %%

        Ответ
        %%(json)
        {"data": {
            "id": 10,
            "author": { пользователь в стандартном формате },
            "body": "Page markup",
            "created_at": "date",
            "title": "title of page"
        }}
        %%

        """
        page = self.request.page
        revision = get_object_or_404(page.revision_set, id=revision_id)
        return Response(RevisionSerializer(revision, context={'request': self.request}).data)
