from django.http import Http404
from wiki.intranet.models import Staff

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.framework import Response, WikiAPIView, raises
from wiki.api_frontend.serializers.staff import StaffSerializer
from wiki.users.dao import staff_model_by_login


class StaffView(WikiAPIView):
    """
    View для получения данных про staff.
    """

    @raises(InvalidDataSentError)
    def get(self, request, *args, **kwargs):
        """
        Вернуть данные про staff.

        %%login%% - стаффовский логин
        %%lang%% - язык передаваемых имени и фамилии, если параметр не указать, то будет использован русский.

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/.staff?login=vpupkin&lang=en"
        %%

        Пример ответа:

        %%(json)
        {
            first_name: "Vasya",
            last_name: "Pupkin",
            is_dismissed: false
        }
        %%
        """
        if 'login' not in request.GET:
            raise InvalidDataSentError('You must specify "login" request parameter')

        login = request.GET['login']
        try:
            staff = staff_model_by_login(login)
        except Staff.DoesNotExist:
            raise Http404('No staff with login=' + login)

        lang = request.GET['lang'] if 'lang' in request.GET else None
        context = {'lang': lang} if lang else None

        return Response(StaffSerializer(staff, context=context).data)
