
from rest_framework.response import Response

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.framework import PageAPIView
from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.subpages import SubpagesRequestDeserializer
from wiki.pages.logic.subpages import get_subpages_tags


class SubpagesView(PageAPIView):
    available_for_admins = True

    @raises()
    def get(self, request, *args, **kwargs):
        """
        Вернуть список URL страниц из кластера.

        Учитываются все страницы, в том числе и закрытые.

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/some/cluster/.subpages?owner=!"
        %%

        Параметры:
        %%owner%% - необязательный параметр, если задан, то возвращаются только
        URL страниц с данными автором. В качестве значения на данный момент
        поддерживается только "!" – авторы корневой страницы кластера.

        Пример ответа:

        %%(json)
        {
            "urls": [
                "/Некий/Кластер/Паге1",
                "/Некий/Кластер/Паге1/Сабпаге",
                "/Некий/Кластер/Паге2",
                ...
            ],
            "limit_exceeded": False,
        }
        %%

        %%limit_exceeded%% равен True, если число страниц в кластере,
        удовлетворяющих запросу, оказалось больше 5000.
        """

        deserializer = SubpagesRequestDeserializer(data=request.GET)
        if not deserializer.is_valid():
            raise InvalidDataSentError(deserializer.errors)

        get_params = deserializer.validated_data

        authors = None
        if get_params.get('owner') == '!':
            authors = request.page.get_authors()

        tags, limit_exceeded = get_subpages_tags(root_supertag=request.page.supertag, authors=authors)

        return Response(
            {
                'urls': ['/' + tag for tag in tags],
                'limit_exceeded': limit_exceeded,
            }
        )
