import logging

from wiki.api_core.errors.rest_api_error import WikiFormatterError
from wiki.api_core.framework import PageAPIView, Response
from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.pages import PageTableOfContentSingleSerializer

logger = logging.getLogger(__name__)


class PageToc(PageAPIView):
    """
    View для получения оглавления страницы.
    """

    @raises(WikiFormatterError)
    def get(self, request, *args, **kwargs):
        """
        Получить оглавление страницы.

        Пример запроса:

        %%
        GET /_api/frontend/vasya/somepage/.toc
        %%

        Пример ответа:

        %%(js)
        {
            "debug": {
                ....
            },
            "data": {
                "toc": {
                    "content": [
                        {
                            "content": [
                                {
                                    "wiki-attrs": {
                                        "txt": "item1",
                                        "anchor": "item1",
                                        "level": 1
                                    },
                                    "block": "wiki-tocitem"
                                },
                                {
                                    "wiki-attrs": {
                                        "txt": "item21",
                                        "anchor": "item21",
                                        "level": 2
                                    },
                                    "block": "wiki-tocitem"
                                },
                                {
                                    "wiki-attrs": {
                                        "txt": "item22",
                                        "anchor": "item22",
                                        "level": 2
                                    },
                                    "block": "wiki-tocitem"
                                }
                            ],
                            "wiki-attrs": {
                                "page": "/vasya/somepage"
                            },
                            "block": "wiki-toc"
                        }
                    ],
                    "wiki-attrs": {
                        "lang": "ru",
                        "path": "/vasya/somepage",
                        "mode": "view"
                    },
                    "block": "wiki-doc"
                }
            },
            "user": {
                ....
            }
        }
        %%

        """

        return Response(PageTableOfContentSingleSerializer(self.request.page, context={'request': self.request}).data)
