
import logging
import re

from django.core.cache import caches

from wiki.legacy.translate import Translator, UnknownTranslateDirection
from wiki.api_core.deprecator import deprecated_api
from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.framework import PageAPIView
from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.grids.grid import GridTranslationSerializer
from wiki.api_frontend.views.grids import GridDataView
from wiki.pages.models import Page
from wiki.utils.timer import track_time

logger = logging.getLogger(__name__)

# значение timeout для кэширования динамического содержимого страницы в секундах
CACHE_TIMEOUT_FOR_DYNAMIC_CONTENT = 60 * 15

# List of things that should NEVER be translated by Yandex.Translate:
# Order is somewhat significant

translator = Translator()

# Everything between special "notranslate" markers
translator.prepend_escape_pattern(re.compile(r'<!--\s*notranslate\s*-->.*?<!--\s*/notranslate\s*-->', re.S))

#    Everything inside <pre>
translator.prepend_escape_pattern(re.compile(r'<pre.*?>.*?<\/pre>', re.S))

#    Inline code: %%(code inline) ... %%
translator.prepend_escape_pattern(re.compile(r'<code[^>]*>.*?</code>'))

# ...Here we cut any html tags with default escape regexp and proceed:

#    Logins and maillists as in tools@
translator.append_escape_pattern(re.compile(r'[\w-]+@'))

#    Slashes in wiki links /page/subpage
translator.append_escape_pattern(re.compile(r'&nbsp;/&nbsp;'))

#    Backslashes as in \\miracle\fish\whale
translator.append_escape_pattern(re.compile(r'\\\\([\w.:-]+\\?)*'))

#    Html entities: &mdash;, &amp; etc
translator.append_escape_pattern(re.compile(r'&\w+;'))


class TranslateView(PageAPIView):
    """
    Ручка для работы с переводами табличных списков.

    WARNING: На данный момент не переводит содержимое, только структуру
    Удалить, после того, как функционал перевода гридов
    переедет на фронт WIKI-14416

    GET-параметр tr_lang должен быть вида "locale1-locale2",
    где localeX - 2 символа, обозначающие локаль.

      * locale1 - с какого языка переводить,
      * locale2 - на какой язык переводить.

    """

    @raises()
    @deprecated_api
    def get(self, request, tr_lang, *args, **kwargs):
        """
        Получить перевод табличного списка.

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.translate/<from_lang>-<to_lang>"
        %%

        Предполагается что пользователь знает на каком языке написан контент
        <from_lang> - с какого языка переводить
        <to_lang> - на какой язык переводить

        Ответ будет выглядеть так:
        %%(json)
        {
          "structure": {
            "sorting": [
              {
                type: "asc",
                name: "2"
              }
            ],
            "title": "grid",
            "fields": [
              {
                "title": "Камент",
                "required": false,
                "type": "string",
                "sorting": "asc",
                "name": "2"
              },
            "done": false,
          },
          "rows": [
            [
              {
                "sort": "123 456",
                "raw": "123 456 ",
                "b64wom1": "...",
                "transformed": "...",
                "row_id": "7",
                "__key__": "2"
              }
            ]
          ],
          "version": "6005994"
        }
        %%
        """
        page = self.request.page
        if not page.page_type == Page.TYPES.GRID:
            raise InvalidDataSentError("Only for grids")

        translation = CachedTranslation(page, tr_lang)
        lang_from, lang_to = tr_lang.split('-')

        data = translation.get_data()
        if data is not None:
            title, rows, structure = data
            page.title = title

            page.rows = rows
            page.structure = structure
        else:
            pageView = GridDataView()

            pageView.request = request
            response = pageView.get(request, *args, **kwargs)

            if response.status_code != 200:
                return response

            timeout = 0
            self.translate_grid(page, response.data, tr_lang)
            # поместить перевод в кэш
            translation.set_data(page.title, page.rows, page.structure, timeout)

        page.lang = lang_to

        return self.build_response(
            instance=page,
            context={
                'user': self.request.user,
                'page': self.request.page,
            })

    def get_serializer_class(self):
        """
        Вернуть сериализатор страницы

        @rtype: GridTranslationSerializer
        """
        return GridTranslationSerializer

    @staticmethod
    def get_translation(pieces, tr_lang):
        """
        Осуществить перевод отдельных кусков текста, собранных в список.
        Формат текста может быть как plain text, так и html.

        @param pieces: список строковых значений для перевода
        @param lang_direction: направление перевода (например, "ru-en")
        @rtype: list
        """
        try:
            return translator(text=pieces, lang=tr_lang, srv='wiki')
        except UnknownTranslateDirection:
            msg = 'Incorrect value of translation direction: "%s"'
            logger.error(msg, tr_lang)
            raise InvalidDataSentError({'lang': msg % tr_lang})

    def translate_grid(self, page, data, tr_lang):
        """
        Осуществить перевод табличного списка

        @param page: объект страницы
        @param data: ответ, полученный от GET запроса страницы
        @param tr_lang: направление перевода (например, "ru-en")
        """

        structure = data['structure']
        list_for_transl = [structure['title']]
        for field in structure['fields']:
            # собрать для перевода значения имен столбцов
            list_for_transl.append(field['title'])

        # получить перевод
        transl_result = self.get_translation(list_for_transl, tr_lang)

        # заменить title страницы переведенным значением
        page.title = transl_result.pop(0)
        structure['title'] = page.title

        for field in structure['fields']:
            # заменить имена столбцов переведенными значениями
            field['title'] = transl_result.pop(0)
        page.structure = structure

        page.rows = data['rows']


def build_cache_key(id, lang):
    """
    Получить ключ для переводов.

    @type id: int
    @type lang: basestring
    @return: str
    """
    return 'translation:{0}:{1}'.format(id, lang)


class CachedTranslation(object):
    """Translation of wiki page stored in cache."""

    def __init__(self, page, translation_lang):
        self.page = page
        self.key = build_cache_key(page.id, translation_lang)
        self.version = page.modified_at
        self.cache = caches['translated_pages_2']

    @track_time
    def get_data(self):
        """
        Вернуть из кэша сериализованные объекты,
        содержащие перевод различных частей страницы.
        """
        return self.cache.get(self.key, version=self.version)

    @track_time
    def set_data(self, title, rows, structure, timeout=0):
        """
        Поместить в кэш сериализованные объекты, содержащие перевод различных частей страницы.
        Если timeout не задан, то в кэше используется значение timeout, полученное при инициализации кэша.

        @type title: basestring
        @type rows: basestring
        @type structure: basestring
        @type timeout: int
        """
        if timeout:
            self.cache.set(self.key, (title, rows, structure), timeout, version=self.version)
        else:
            self.cache.set(self.key, (title, rows, structure), version=self.version)
