import logging
from datetime import datetime

from django import forms
from django.db.models import Q
from model_utils import Choices
from rest_framework.response import Response

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.errors.permissions import UserHasNoAccess
from wiki.api_core.framework import PageAPIView
from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.tree_cluster import TreeClusterSerializer
from wiki.org import get_org
from wiki.pages.access import ACCESS_DENIED, get_bulk_access_status
from wiki.pages.models import Page
from wiki.utils.timezone import make_aware_utc

logger = logging.getLogger(__name__)

PAGE_TYPES = Choices(
    ('P', 'page', 'page'),
    ('G', 'grid', 'grid'),
    ('R', 'redirect', 'redirect'),
    ('403', 'access_denied', 'access denied'),
    ('404', 'not_found', 'not found'),
)


class MultipleChoiceSplittedField(forms.MultipleChoiceField):
    SPLIT_BY = ','

    def to_python(self, value):
        if not value:
            return []
        # потому что АПИ спроектировано так, что принимает page_types через запятую
        # надо спроектировать так, чтобы было понятно фреймворку что page_types - массив.
        # вместо /handle?page_types=foo,bar
        # надо сделать так: /handle?page_types=foo&page_types=bar
        result = []
        for item in value:
            result.extend(item.split(self.SPLIT_BY))
        return super(MultipleChoiceSplittedField, self).to_python(result)


class TreeClusterParamsValidator(forms.Form):
    sort_order = forms.ChoiceField(choices=Choices('tag', 'title', 'modified_at', 'created_at'), required=True)

    page_types = MultipleChoiceSplittedField(choices=PAGE_TYPES, required=False)

    def clean_or_raise(self):
        if self.is_valid():
            return self.cleaned_data

        raise InvalidDataSentError(self.errors)


class TreeClusterView(PageAPIView):
    """
    View для работы с деревом кластера в сайдбаре.
    """

    serializer_class = TreeClusterSerializer

    def check_page_access(self):
        """
        Права доступа к текущей странице не проверяем, так как сама страница не отображается, а рассматриваются
        только ее дочерние страницы.
        """
        if not self.request.user.is_authenticated:
            raise UserHasNoAccess

    @raises()
    def get(self, request, *args, **kwargs):
        """
        Получить дерево кластера как список дочерних страниц.

        Параметры:
        #|
        || **имя** | **тип** | **обязательность** | **описание** ||
        || %%page_types%% | str | не обязательный | список типов возвращаемых страниц (кроме типа 'P', так как он
        включается в результат всегда).
        Если параметр отсутствует, то возвращаются только обычные страницы с типом 'P' ||
        || %%sort_order%% | str | !!обязательный!! | имя атрибута, по значению которого необходимо отсортировать
        возвращаемые данные ||
        |#

        Возвращает: json со списом атрибутов страниц.

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/homepage/.treecluster?sort_order=tag&page_types=G,R,403,404"
        %%

        Пример ответа:
        %%(json)
        {
            "debug": {
                ....                                        // дебаг информация, в продакшне может быть отключена
            },
            "data": [
                {
                    "supertag": "homepage/wiki",
                    "tag": "ХомеПаге/вики",
                    "url": "/HomePage/wiki",
                    "title": "Normal wiki page",
                    "type": "P",
                    "children_count": 3,
                    "created_at": "2013-04-03T17:08:40+03:00",
                    "modified_at": "2013-04-03T17:08:40+03:00"
                },
                {
                    "supertag": "homepage/grid",
                    "tag": "ХомеПаге/грид",
                    "url": "/HomePage/grid",
                    "title": "Wiki Grid",
                    "type": "G",
                    "children_count": 0,
                    "created_at": "2013-04-03T17:08:40+03:00",
                    "modified_at": "2013-04-03T17:08:40+03:00"
                },
                {
                    "supertag": "homepage/redirect",
                    "tag": "ХомеПаге/редирект",
                    "url": "/HomePage/redirect",
                    "title": "REDIRECT page",
                    "type": "R",
                    "children_count": 0,
                    "created_at": "2013-04-03T17:08:40+03:00",
                    "modified_at": "2013-04-03T17:08:40+03:00"
                }
                {
                    "supertag": "homepage/limited",
                    "tag": "ХомеПаге/лимитед",
                    "url": "/HomePage/limited",
                    "title": null,
                    "type": "403",
                    "children_count": 15,
                    "created_at": "2013-04-03T17:08:40+03:00",
                    "modified_at": "2013-04-03T17:08:40+03:00"
                }
                {
                    "supertag": "homepage/not_found",
                    "tag": "ХомеПаге/Нот_Фоунд",
                    "url": "/HomePage/Not_Found",
                    "title": null,
                    "type": "404",
                    "children_count": 10,
                    "created_at": null,
                    "modified_at": null
                }
            ],
        }
        %%

        Если обязательный параметр %%sort_order%% отсутствует, то в ответе будет передана
        ошибка с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": "Client sent invalid data",
                "errors": {
                    'sort_order': 'Mandatory parameter'
                },
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%
        """
        if request.page:
            supertag = request.page.supertag
        else:
            supertag = request.supertag

        validator = TreeClusterParamsValidator(request.GET)
        data = validator.clean_or_raise()

        return Response(
            self.serializer_class(
                get_tree_cluster_data(
                    root_supertag=supertag,
                    user=request.user,
                    sort_order=data['sort_order'],
                    page_types=data.get('page_types'),
                ),
                many=True,
            ).data
        )


LONG_TIME_AGO = make_aware_utc(datetime(1, 1, 1))


def get_tree_cluster_data(root_supertag, user, sort_order, page_types=None):
    """
    Построить дерево кластера для страницы с переданным supertag.
    Дерево кластера представляет из себя список дочерних страниц первого уровня с указанием для каждой такой страницы
    количества дочерних страниц 2-го уровня (относительно страницы с переданным supertag)

    @type root_supertag: basestring
    @param root_supertag: корневая страница кластера
    @type user: django user
    @param user: пользователь, запрашивающий дерево кластера. Для этого пользователя будут определяться права доступа
    к страница кластера
    @type sort_order: basestring
    @param sort_order: имя атрибута для сортировки результата
    из списка ('tag', 'title', 'modified_at', 'created_at')
    @type page_types: set
    @param page_types: список имен типов страниц, отображаемых в дереве кластера дополнительно
    к простым страницам с типом 'P'.
    Возможные варианты: ('G', 'R', '403', '404').
    @rtype list
    @return список, где каждый элемент это словарь с ключами 'tag', 'url', 'title', 'page_type', 'children_count',
    'modified_at', 'created_at'
    """
    additional_return_page_types = page_types or {}
    view_404_pages = PAGE_TYPES.not_found in additional_return_page_types

    excluded_types = []
    if PAGE_TYPES.redirect not in additional_return_page_types:
        excluded_types.append(PAGE_TYPES.redirect)

    if PAGE_TYPES.grid not in additional_return_page_types:
        excluded_types.append(PAGE_TYPES.grid)

    # выбираем все подстраницы, потому что нет эффективного способа запросить
    # только страницы 1-го и 2-го уровня
    query_set = Page.active.filter(supertag__startswith='%s/' % root_supertag, org=get_org())

    if excluded_types and not view_404_pages:
        # если не надо искать несуществующие страницы, то фильтруем данные сразу на уровне запроса,
        # иначе надо фильтровать чуть позднее, чтобы отфильтрованные на этом этапе страницы не попали
        # в число несуществующих

        if excluded_types == [PAGE_TYPES.grid]:
            # Эту ситуацию нужно рассмотреть отдельно.
            # Мы должны исключить страницы с типом Грид, но при этом не исключать редиректы.
            query_set = query_set.filter(~Q(page_type=Page.TYPES.GRID) | Q(redirects_to__isnull=False))
        else:
            # Сюда мы попадаем, если выбрано показывать только гриды, или
            # если выбрано не показывать ни гриды, ни редиректы.
            # Поэтому редиректы можно исключать безусловно.
            if PAGE_TYPES.grid in excluded_types:
                query_set = query_set.exclude(page_type=Page.TYPES.GRID)
            query_set = query_set.filter(redirects_to__isnull=True)

    data = query_set.values_list('supertag', 'tag', 'title', 'modified_at', 'created_at', 'page_type', 'redirects_to')

    # словарь, где ключ - supertag страницы первого уровня,
    # а значение - атрибуты страницы с этим supertag
    pages_attrs_level_1 = {}
    pages_supertags_level_1 = []
    tags_level_2_plus = []
    excluded_supertags = []

    root_supertag_with_slash_length = len(root_supertag) + len('/')
    for supertag, tag, title, modified_at, created_at, page_type, redirects_to in data:
        breadcrumbs = supertag[root_supertag_with_slash_length:].split('/')
        breadcrumbs_length = len(breadcrumbs)

        if breadcrumbs_length == 1:
            if page_type not in (Page.TYPES.PAGE, Page.TYPES.WYSIWYG, Page.TYPES.GRID):
                page_type = PAGE_TYPES.page

            if redirects_to is not None:
                page_type = PAGE_TYPES.redirect

            if view_404_pages and page_type in excluded_types:
                excluded_supertags.append(supertag)
            else:
                pages_supertags_level_1.append(supertag)

            pages_attrs_level_1[supertag] = {
                'tag': tag,
                'url': '/' + supertag,
                'title': title,
                'modified_at': modified_at,
                'created_at': created_at,
                'page_type': page_type,
                'children_count': 0,
            }
        elif breadcrumbs_length == 2 or view_404_pages:
            tags_level_2_plus.append((breadcrumbs[0], breadcrumbs_length))

    for supertag, level in tags_level_2_plus:
        try:
            pages_attrs_level_1[root_supertag + '/' + supertag]['children_count'] += 1 if level == 2 else 0
        except KeyError:
            if view_404_pages:
                # на первом уровне несуществующая страница. Добавим ее в наш словарь
                page_level_1_supertag = root_supertag + '/' + supertag
                pages_attrs_level_1[page_level_1_supertag] = {
                    'tag': page_level_1_supertag,
                    'url': '/' + page_level_1_supertag,
                    'title': None,
                    'children_count': 1,
                    'page_type': PAGE_TYPES.not_found,
                    'modified_at': None,
                    'created_at': None,
                }

    # фильтрация данных для случая, когда этого нельзя было делать в запросе
    if view_404_pages and excluded_types and excluded_supertags:
        for excluded_supertag in excluded_supertags:
            del pages_attrs_level_1[excluded_supertag]

    # проверим права доступа пользователя к страницам первого уровня
    access_statuses = get_bulk_access_status(pages_supertags_level_1, user)
    for root_supertag, status in access_statuses.items():
        if status == ACCESS_DENIED:
            if PAGE_TYPES.access_denied in additional_return_page_types:
                pages_attrs_level_1[root_supertag]['title'] = None
                pages_attrs_level_1[root_supertag]['page_type'] = PAGE_TYPES.access_denied
            else:
                del pages_attrs_level_1[root_supertag]

    tree_cluster = list(pages_attrs_level_1.values())

    # сортировка результата
    if sort_order == 'modified_at' or sort_order == 'created_at':
        tree_cluster.sort(key=lambda x: x[sort_order] or LONG_TIME_AGO, reverse=True)
    else:
        tree_cluster.sort(key=lambda x: x[sort_order].lower() if x[sort_order] else '')

    return tree_cluster
