
from rest_framework.response import Response

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.framework import PageAPIView
from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.tutorial import TutorialGetSerializer, TutorialPostSerializer
from wiki.users.logic.tutorial import get_next_tutorial_step, set_tutorial_stage_completed, start_tutorial_again


class TutorialView(PageAPIView):
    serializer_class = TutorialPostSerializer

    @raises()
    def get(self, request, *args, **kwargs):
        """
        Возвращает очередную стадию туториала.

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<page_supertag>/.tutorial?entry_point=view"
        %%

        Возможен один из трех результатов:

        a) Отобразить стадию с именем stage

        %%(js)
        {
            "data": {
                "status": "display_stage",
                "stage": "promo",
            },
            "debug": {...},
            "user": {...}
        }
        %%

        b) В данный момент ничего отображать не надо

        %%(js)
        {
            "data": {
                "status": "nothing_to_display",
            },
            "debug": {...},
            "user": {...}
        }
        %%

        c) Пользователь прошел весь туториал

        %%(js)
        {
            "data": {
                "status": "tutorial_completed",
            },
            "debug": {...},
            "user": {...}
        }
        %%

        """
        deserializer = TutorialGetSerializer(data=request.GET)

        if not deserializer.is_valid():
            raise InvalidDataSentError(deserializer.errors)

        get_params = deserializer.validated_data

        return Response(
            get_next_tutorial_step(
                user=request.user, page=request.page, entry_point=get_params['entry_point']
            ).as_json()
        )

    @raises()
    def post(self, request, *args, **kwargs):
        """
        Запоминаем, что пользователь прошел указанную стадию туториала.

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" -XPOST \
        "https://wiki-api.yandex-team.ru/_api/frontend/<page_supertag>/.tutorial"
        %%

        Тело запроса:

        %%(js)
        {
            "stage": "promo"
        }
        %%

        """

        data = self.validate()

        set_tutorial_stage_completed(user=request.user, stage=data['stage'])

        return self.build_success_status_response()

    @raises()
    def delete(self, request, *args, **kwargs):
        """
        Начать туториал заново.

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" -XDELETE \
        "https://wiki-api.yandex-team.ru/_api/frontend/<page_supertag>/.tutorial"
        %%

        """
        start_tutorial_again(user=request.user)

        return self.build_success_status_response()
