from django.conf import settings as django_settings
from django.db import transaction
from django.http import HttpResponseForbidden
from django.utils.translation import ugettext_lazy as _

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.framework import WikiAPIView
from wiki.api_core.raises import raises
from wiki.api_core.utils import is_tvm_authentication
from wiki.api_frontend.serializers.settings import SettingsSerializer
from wiki.users.logic import settings as settings_logic, set_user_lang_ui
from wiki.utils import staff


class SettingsView(WikiAPIView):
    """
    Получение и изменение настроек пользователя.
    """

    serializer_class = SettingsSerializer

    @raises()
    def get(self, request, **kwargs):
        """
        Получить значение настроек пользователя.

        %%
        curl -H "Authorization: OAuth <token>" -XGET \
        "https://wiki-api.yandex-team.ru/_api/frontend/.settings"
        %%
        Тело запроса пустое.

        Тело ответа:
        %%(js)
        {
            "language": "en",
            "code_theme": "dark",
            "propose_content_translation": true,
            "use_full_width_content": false,
            "use_new_wf": false
        }
        %%
        """
        settings = settings_logic.get_user_settings(request.user)
        if django_settings.IS_INTRANET:
            settings['language'] = request.LANGUAGE_CODE or 'ru'
        return self.build_response(instance=settings)

    @raises(InvalidDataSentError)
    @transaction.atomic
    def post(self, request, **kwargs):
        """
        Изменить значение настроек. Можно присылать любое количество
        настроек — изменены будут только присланные.

        %%
        curl -H "Authorization: OAuth <token>" \
        -XPOST "https://wiki-api.yandex-team.ru/_api/frontend/.settings/"
        %%
        Тело запроса
        %%(js)
        {
            "language": "ru",
            "code_theme": "dark"
        }
        %%

        Если нет ошибок, тело ответа будет пустым.
        """
        user = request.user

        data = self.validate()

        # смена языка — это запрос в стафф, так что обрабатываем отдельно
        if django_settings.IS_INTRANET and 'language' in data:
            # для смены языка нам нужен пользовательский тикет
            if not (is_tvm_authentication(request) and request.user_auth.tvm2_user_ticket):
                return HttpResponseForbidden(_('tvm2 user ticket is required'))

            language = data.pop('language')
            staff.change_user_language(
                user=user,
                language=language,
                user_auth=request.user_auth,
            )

            # сохраняем язык сразу в своей базе, так ждать обновления из стаффа - это долго.
            set_user_lang_ui(user, language)

        if data:
            settings_logic.update_user_settings(user, data)

        return self.build_success_status_response()
