import logging

from django.db.models import Q
from django_pgaas import atomic_retry
from rest_framework.response import Response

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.framework import PageAPIView
from wiki.api_core.raises import raises
from wiki.api_core.serializers import UserSerializer
from wiki.api_frontend.serializers.io import EmptySerializer
from wiki.api_frontend.serializers.watch import MassWatchInputDataSerializer, WatchResultSerializer
from wiki.api_v2.exceptions import AlreadyExists
from wiki.favorites_v2.tasks import UpdateWatcherAutofolderTask
from wiki.org import get_org, get_org_id
from wiki.pages.logic import subscription
from wiki.pages.logic.subscription import get_united_subscribed_users
from wiki.pages.models import PageWatch
from wiki.subscriptions import logic as new_subscriptions
from wiki.subscriptions.models import SubscriptionType
from wiki.users.logic.settings import uses_new_subscriptions
from wiki.utils.diff import DifflibDiff
from wiki.utils.subscription import generate_event
from wiki.utils.db import on_commit

logger = logging.getLogger(__name__)

_diff = DifflibDiff(return_type='chunks')


class WatchView(PageAPIView):
    """
    Подписка на страницу.
    """

    serializer_class = EmptySerializer

    @raises()
    def post(self, request, *args, **kwargs):
        """
        Подписать текущего пользователя на страницу.

        Параметры:
        #|
        || **имя** | **тип** | **обязательность** | **описание** ||
        || %%unwatch_children%% | boolean | не обязательный | флаг, разрешающий перед подпиской на страницу отписать
        пользователя от страничного кластера, если он был подписан на текущую страницу как на кластер ||
        |#

        Пример запроса:
        %%
        curl -X "POST" -H "Authorization: OAuth <token>" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.watch"
        %%

        В случае успешного запроса возвращает количество страниц, подписка на которые
        поменялась, т.е. единицу
        При повторном запросе всегда будет 0, если между запросами ничего не изменялось.

        Пример ответа:
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "data": {
                "pages_count": 1            // Количество страниц, подписка на которые поменялась.
            }
        }
        %%
        """
        page = self.request.page
        user = self.request.user
        username = user.username
        pages_count = 0
        need_subscribe = False

        if uses_new_subscriptions(user):
            try:
                new_subscriptions.create_subscription(user=user, page=page, type_=SubscriptionType.MY, is_cluster=False)
                generate_event('watch', user, page)
                pages_count = 1
            except AlreadyExists:
                pass

            return Response(WatchResultSerializer({'pages_count': pages_count}).data)

        if (
            self._get_bool(self.request, 'unwatch_children')
            and self.request.page.pagewatch_set.filter(user=username, is_cluster=True).exists()
        ):
            MassUnwatchView.unsubscribe(self.request)
            need_subscribe = True

        if need_subscribe or not self.request.page.pagewatch_set.filter(user=username).exists():
            subscription.create_watch(page=page, user=user, is_cluster=False)
            generate_event('watch', user, page)
            pages_count = 1

            if not self.request.user.is_robot():
                # обновляем закладки в автопапке 'Я наблюдатель'
                update_task = UpdateWatcherAutofolderTask()
                on_commit(lambda: update_task.delay(user_id=user.id, username=username, org_id=get_org_id()))

        return Response(WatchResultSerializer({'pages_count': pages_count}).data)


class MassWatchView(PageAPIView):
    """
    Подписка на кластер.
    """

    serializer_class = MassWatchInputDataSerializer

    @raises()
    @atomic_retry()
    def post(self, request, *args, **kwargs):
        """
        Подписать одного или нескольких пользователей на кластер страниц.

        Параметры:
        #|
        || **имя** | **тип** | **обязательность** | **описание** ||
        || %%uids%% | list | не обязательный | список UID пользователей, которых надо подписать.
        Если параметр отсутствует, то подписывается только один пользователь, отправивший запрос ||
        || %%comment%% | str | не обязательный | Текст комментария, который будет отправлен в письме пользователю
        о подписке ||
        |#

        Пример запроса:
        %%
        curl -X "POST" -H "Authorization: OAuth <token>" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.masswatch"
        %%

        Тело запроса:
        %%(js)
        {
            "uids": [10605, 9607, 12345],
            "comment": "Я тебя подписал на нашу страницу"
        }
        %%

        Возвращает: количество страниц, подписка на которые поменялась.
        При повторном запросе всегда будет 0, если между запросами ничего не изменялось.

        Пример ответа:
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "data": {
                "pages_count": 1            // Количество страниц, подписка на которые поменялась.
            }
        }
        %%

        Если превышено максимально допустимое число пользователей для подписки в одном запросе,
        равное MAX_USERS_NUMBER=100, то в ответе будет передана ошибка с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": "Client sent invalid data",
                "errors": {
                    'uids': ['The maximum number of users in the query is exceeded: 100']
                },
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }

        """
        serializer = self.get_serializer(data=request.data)

        if serializer.is_valid():
            pages_count = serializer.save(self.request.user, self.request.page)
            return Response(WatchResultSerializer({'pages_count': pages_count}).data)
        else:
            raise InvalidDataSentError(serializer.errors)


class UnwatchView(PageAPIView):
    """
    Отписка от страницы.
    """

    serializer_class = EmptySerializer

    @raises()
    def post(self, request, *args, **kwargs):
        """
        Отписать текущего пользователя от страницы.

        Пример запроса:
        %%
        curl -X "POST" -H "Authorization: OAuth <token>" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.unwatch"
        %%

        В случае успешного запроса будет возвращен пустой 200-ый ответ.
        """
        user = self.request.user
        page = self.request.page
        if uses_new_subscriptions(user):
            new_subscriptions.remove_current_subscriptions([user], page)
            generate_event('unwatch', user, page)
            return Response(WatchResultSerializer({'pages_count': 1}).data)

        queryset = page.pagewatch_set.filter(user=user.username)
        pages_count = 0
        if queryset.exists():
            queryset.delete()
            generate_event('unwatch', user, page)
            pages_count = 1

            if not self.request.user.is_robot():
                # обновляем закладки в автопапке 'Я наблюдатель'
                update_task = UpdateWatcherAutofolderTask()
                on_commit(lambda: update_task.delay(user_id=user.id, username=user.username, org_id=get_org_id()))

        return Response(WatchResultSerializer({'pages_count': pages_count}).data)


class MassUnwatchView(PageAPIView):
    """
    Отписка от кластера.
    """

    serializer_class = EmptySerializer

    @raises()
    def post(self, request, tag, *args, **kwargs):
        """
        Отписаться от всего кластера (страницы и ее подстраниц).

        %%(sh)
        curl -H "Authorization: OAuth <token>" -X "POST" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.massunwatch"
        %%
        Ответ:
        %%(json)
        {"data": {
            "pages_count": 10
        }}
        %%

        pages_count - от скольки страниц был отписан пользователь.
        """
        pages_count = self.unsubscribe(self.request)
        return Response(WatchResultSerializer({'pages_count': pages_count}).data)

    @staticmethod
    def unsubscribe(request):
        if uses_new_subscriptions(request.user):
            if user_subscription := new_subscriptions.get_user_page_subscriptions(request.user, request.page):
                user_subscription.delete()
                generate_event('unwatch', request.user, request.page, cluster=[request.page.supertag])
            return 1

        queryset = PageWatch.objects.filter(
            Q(page__supertag=request.supertag, page__org=get_org())
            | Q(page__supertag__startswith=request.supertag + '/', page__org=get_org())
        )
        queryset = queryset.filter(user=request.user.username)
        queryset = queryset.only('page__supertag')
        # TODO: запрос queryset.delete унести под if - убрать лишний запрос
        supertags = [subscription.page.supertag for subscription in queryset]
        queryset.delete()
        if len(supertags) > 0:
            generate_event('unwatch', request.user, request.page, cluster=supertags)
            if not request.user.is_robot():
                # обновляем закладки в автопапке 'Я наблюдатель'
                update_task = UpdateWatcherAutofolderTask()
                on_commit(
                    lambda: update_task.delay(
                        user_id=request.user.id, username=request.user.username, org_id=get_org_id()
                    )
                )

        return len(supertags)


class WatchersView(PageAPIView):
    """
    Список подписчиков страницы.
    """

    @raises()
    def get(self, request, *args, **kwargs):
        """
        Вернуть список подписчиков страницы.

        Пример запроса:
        %%
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.watchers"
        %%

        Пример ответа:
        %%(json)
        [
            {
                "uid": 1234,
                "login": "v-pupkin",
                "first_name": "Вася",
                "last_name": "Пупкин",
                "display": "Вася Пупкин",
                "email": "v-pupkin@yandex-team.ru",
                "is_admin": false,
                "is_external_employee": false,
                "is_dismissed": false
            },
            ...
        ]
        %%

        """

        users = get_united_subscribed_users(request.page)
        watchers_data = [UserSerializer(watcher).data for watcher in users]
        return Response(watchers_data)
