
from django.conf import settings

from wiki.sync.connect.logic import is_org_imported
from wiki.sync.connect.models import Organization, SyncStatistics
from wiki.org import org_ctx, org_user, get_user_orgs
from wiki.sync.connect.dir_client import DirClient

WIKI_EVENT_NAMES = [
    'service_enabled',
    'service_disabled',
    'user_added',
    'department_added',
    'group_added',
    'user_dismissed',
    'user_property_changed',
    'user_moved',
    'department_property_changed',
    'department_moved',
    'department_deleted',
    'group_deleted',
    'group_property_changed',
    'group_membership_changed',
    'domain_master_changed',
    'organization_subscription_plan_changed',
    'organization_deleted',
]


class AlreadySubscribedError(Exception):
    pass


def delete_org_data(dir_org_id):
    # User удаляем отдельно, т.к. они M-to-N к Organization.
    org = Organization.objects.get(dir_id=dir_org_id)
    with org_ctx(org):
        users = org_user()
        for user in users:
            # Если пользователь остался только в одной организации,
            # то его можно удалить.
            if len(get_user_orgs(user)) == 1:
                user.delete()

    # Остальные сущности удалятся каскадно при удалении Organization.
    org.delete()


def subscribe_to_events():
    dir_client = DirClient()

    # проверим, что Wiki еще не подписана
    result = dir_client.get_subscriptions()
    if len(result['result']) > 0:
        raise AlreadySubscribedError()

    data = dir_client.set_subscriptions(
        url='https://{host}/{path}'.format(host=settings.API_WIKI_HOST, path=settings.WIKI_SUBSCRIPTION_PATH),
        event_names=WIKI_EVENT_NAMES,
    )

    return data


def get_failed_syncs(dir_org_id=None):
    dir_client = DirClient()
    not_ready_dir_org_ids = [org['id'] for org in dir_client.get_organizations(**{'service.ready': False})]

    dir_org_ids = set(Organization.objects.all().values_list('dir_id', flat=True))

    sync_stats = SyncStatistics.objects.filter(last_pull_status=SyncStatistics.PULL_STATUSES.failed)
    if dir_org_id:
        sync_stats = sync_stats.filter(dir_org_id=dir_org_id)

    sync_failed = []
    import_failed = []

    for sync in sync_stats:
        dir_org_id = int(sync.dir_org_id)
        if is_org_imported(sync, dir_org_ids):
            sync_failed.append(dir_org_id)
        elif dir_org_id in not_ready_dir_org_ids:
            import_failed.append(dir_org_id)

    return {
        'sync_failed': sync_failed,
        'import_failed': import_failed,
    }
