
import json
import yenv

from django.conf import settings
from django.http import HttpResponse
from rest_framework.generics import GenericAPIView
from rest_framework.permissions import BasePermission

from wiki.sync.connect.tasks import create_new_organization, ProcessChangeEventsTask
from wiki.sync.connect.models import Organization
from wiki.api_internal.logic.dir import delete_org_data, get_failed_syncs, subscribe_to_events
from wiki.org import get_user_orgs
from wiki.sync.connect.dir_client import DirClient


class EtalonOrganizationMembershipPermission(BasePermission):
    """
    Проверяет, что пользователь входит в эталонную организацию Вики в Директории.
    """

    def has_permission(self, request, view):
        if not settings.IS_BUSINESS:
            return False

        if yenv.type in ('development', 'unstable'):
            return True

        orgs = get_user_orgs(request.user)
        return any([int(org.dir_id) == settings.ETALON_ORG_DIR_ID for org in orgs])


class DirApiView(GenericAPIView):
    permission_classes = (EtalonOrganizationMembershipPermission,)
    render_blank_form_for_methods = ()

    def __init__(self, **kwargs):
        super(DirApiView, self).__init__(**kwargs)
        self.dir_client = DirClient()


class OrganizationView(DirApiView):
    def put(self, request, dir_org_id, *args, **kwargs):
        org = self.dir_client.get_organization(dir_org_id)
        if Organization.objects.filter(dir_id=dir_org_id).exists():
            delete_org_data(dir_org_id)
        create_new_organization.delay(org)
        return HttpResponse('ok', status=200)

    def post(self, request, dir_org_id, *args, **kwargs):
        if not Organization.objects.filter(dir_id=dir_org_id).exists():
            return HttpResponse('org not exist', status=400)
        ProcessChangeEventsTask().delay(dir_org_id=str(dir_org_id))
        return HttpResponse('ok', status=200)

    def delete(self, request, dir_org_id, *args, **kwargs):
        delete_org_data(dir_org_id)
        return HttpResponse('ok', status=200)


class SubscribeToEventsView(DirApiView):
    permission_classes = (EtalonOrganizationMembershipPermission,)

    def post(self, request, *args, **kwargs):
        data = subscribe_to_events()

        return HttpResponse(json.dumps(data))


class SubscriptionsListView(DirApiView):
    def get(self, request, *args, **kwargs):
        data = self.dir_client.get_subscriptions()

        return HttpResponse(json.dumps(data))


class SubscriptionView(DirApiView):
    def delete(self, request, subscription_id, *args, **kwargs):
        self.dir_client.delete_subscription(subscription_id)

        return HttpResponse('OK', status=200)


class FailedSyncsView(DirApiView):
    def get(self, request, *args, **kwargs):
        data = get_failed_syncs(request.GET.get('dir_org_id'))

        return HttpResponse(json.dumps(data))
