from typing import List, Dict, Set, Tuple

from wiki.api_core.errors.rest_api_error import ResourceIsMissing
from wiki.api_svc.acl.consts import DcAclType, DcInheritableAcl, DcAcl
from wiki.intranet.models import Staff
from wiki.pages.access import get_bulk_raw_access, interpret_raw_access
from wiki.pages.access.check_access import LegacyAclCache, check_access
from wiki.pages.models import Page, CloudPage


def verify_grantees(supertag: str, staff_ids: List[int]) -> Tuple[Set[int], Set[int]]:
    try:
        page = Page.objects.get(supertag=supertag)
    except Page.DoesNotExist:
        raise ResourceIsMissing()

    staff_models = Staff.objects.filter(id__in=staff_ids).prefetch_related('user').all()
    o = set()
    cache = LegacyAclCache()
    for staff in staff_models:
        if check_access(page, staff.user, staff=staff, cache=cache):
            o.add(staff.id)

    return o, set(staff_ids) - o


def get_acls(supertags: List[str]) -> Dict[str, DcInheritableAcl]:
    pages = Page.objects.filter(supertag__in=supertags)
    cloud_pages: Dict[int, CloudPage] = {
        c.page_id: c for c in CloudPage.objects.filter(page_id__in=[page.id for page in pages])
    }
    raw_accesses = get_bulk_raw_access(pages)
    out = {}
    parent_page = {}

    for page in pages:
        raw_access = raw_accesses[page]
        interpreted_access = interpret_raw_access(raw_access)

        document_id = None

        group_ids = []
        users = []

        try:
            if page.id in cloud_pages:
                document_id = cloud_pages[page.id].get_cloud_src().embedding.sourcedoc

        except (KeyError, TypeError):
            pass

        if interpreted_access['is_common_wiki'] or interpreted_access['is_common']:
            acl_type = DcAclType.DEFAULT
        elif interpreted_access['is_owner']:
            acl_type = DcAclType.ONLY_OWNERS
        elif interpreted_access['is_restricted']:
            acl_type = DcAclType.CUSTOM
            for staff in interpreted_access['users']:
                if staff.is_dismissed:
                    continue
                users.append(staff.id)

            for group in interpreted_access['groups']:
                group_ids.append(group.id)
        else:
            raise ValueError(f'Misconfigured acl for page {page.id}')

        actual_acl_owner = raw_access['latest_supertag']

        if actual_acl_owner is not None and actual_acl_owner != page.supertag:
            if actual_acl_owner not in parent_page:
                parent_page[actual_acl_owner] = Page.objects.prefetch_related('authors', 'authors__staff').get(
                    supertag=actual_acl_owner
                )
            i = DcInheritableAcl(
                acl=DcAcl(
                    supertag=page.supertag,
                    document_id=document_id,
                    acl_type=DcAclType.INHERITED,
                    inherits_from=actual_acl_owner,
                    owners=[u.staff.id for u in page.authors.all()],
                ),
                inherits_from=DcAcl(
                    supertag=actual_acl_owner,
                    acl_type=acl_type,
                    owners=[u.staff.id for u in parent_page[actual_acl_owner].authors.all()],
                    users=users,
                    group_ids=group_ids,
                ),
            )
        else:
            i = DcInheritableAcl(
                acl=DcAcl(
                    supertag=page.supertag,
                    document_id=document_id,
                    acl_type=acl_type,
                    owners=[u.staff.id for u in page.authors.all()],
                    users=users,
                    group_ids=group_ids,
                )
            )

        out[page.supertag] = i
    return out
