from django.conf import settings
from rest_framework import serializers

if settings.IS_BUSINESS:
    from wiki.users.models import GROUP_TYPES


class SuggestRequest(serializers.Serializer):
    text = serializers.CharField(required=True)
    layers = serializers.CharField(default='people')


def get_department(user, org_id):
    user_department = user.groups.filter(group__group_type=GROUP_TYPES.department, group__org_id=org_id).first()
    return user_department.group.title if user_department is not None else ''


class BaseVersionedSerializer(serializers.BaseSerializer):
    def __init__(self, instance=None, version: str = '1', **kwargs):
        self.version = str(version)
        super().__init__(instance, **kwargs)

    def to_representation(self, user):
        if self.version == '2':
            return self._to_representation_v2(user)

        return self._to_representation_v1(user)

    def _to_representation_v1(self, user):
        return {'fields': []}

    def _to_representation_v2(self, user):
        base = self._to_representation_v1(user)
        fields = base.pop('fields')
        for field in fields:
            base[field['type']] = field['value']

        return base


class UserLayerSerializer(BaseVersionedSerializer):
    def _to_representation_v1(self, user):
        return {
            'layer': 'people',
            'title': user.staff.get_full_name(),
            'url': '',
            'id': user.staff.uid,
            'cloud_id': user.cloud_uid,
            'click_urls': [],
            'fields': [
                {'type': 'login', 'value': user.username},
                {'type': 'department_name', 'value': get_department(user, self.context['org_id'])},
                {'type': 'is_dismissed', 'value': user.staff.is_dismissed},
                {'type': 'avatar_url', 'value': 'https://avatars.mds.yandex.net/get-yapic/islands-middle'},
            ],
        }


class GroupLayerSerializer(BaseVersionedSerializer):
    def _to_representation_v1(self, group):
        return {
            'id': group.dir_id,
            'url': '',
            'title': group.title,
            'fields': [],
            'layer': 'groups',
            'click_urls': [],
        }


class DepartmentLayerSerializer(BaseVersionedSerializer):
    def _to_representation_v1(self, department):
        return {
            'id': department.dir_id,
            'url': '',
            'title': department.title,
            'fields': [],
            'layer': 'departments',
            'click_urls': [],
        }
