
import logging
from base64 import b64encode
from io import BytesIO

from rest_framework.response import Response

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.errors.permissions import UserHasNoAccess
from wiki.api_core.raises import raises
from wiki.api_svc.errors import RenderingFailedError
from wiki.api_svc.serializers import GraphFormatterRequest
from wiki.api_svc.views.base import ServiceAPIView
from wiki.legacy.formatters import wf_config
from wiki.legacy.formatters.blockdiag import Blockdiag
from wiki.legacy.formatters.graphviz import Graphviz
from wiki.legacy.formatters.seqdiag import Seqdiag
from wiki.legacy.diag_exceptions import ParseException
from wiki.utils.errors import InputValidationError

logger = logging.getLogger(__name__)

# wf постепенно депрекейтим, поэтому импортирую напрямую без использования конфига.

allowed_formatters = {'blockdiag': Blockdiag, 'seqdiag': Seqdiag, 'graphviz': Graphviz}


class GraphFormatterView(ServiceAPIView):
    """
    View для микросервиса Щщи, рендерит блоки и диаграммы в base64-encoded картинку.
    Параметры

    {
       "output": png|svg, - формат ответа svg или base64-encoded png
       "formatter": blockdiag|seqdiag|graphviz,
       "payload": str,
       "width": int, optional - только для png
       "height": int, optional - только для png
    }

    Ответ об успешной операции:

    {
        "data": {
            "image_source": "str"    // BASE64-encoded сорц изображения
        },
        "user": {
            ....                     // пользователь, сделавший запрос
        },
        "debug": {
            ....                     // дебаг информация, в продакшне может быть отключена
        }
    }

    @todo собирать метрики использования и время отклика - есть предложение вынести данную работу в микросервис
    """

    serializer_class = GraphFormatterRequest

    @raises(UserHasNoAccess, InvalidDataSentError, RenderingFailedError)
    def post(self, request, *args, **kwargs):
        data = self.validate()

        formatter = allowed_formatters[data['formatter']](params={'output': data['output']}, wf_config=wf_config.config)

        try:
            result, stderr = formatter.do_format(data['payload'])
            formatter.parse_stderr_from_tool(stderr)
        except (RuntimeError, AttributeError, ParseException, InputValidationError) as e:
            # Некоторые ошибки, что могут стрелять из потрохов blockdiag/etc
            # AttributeError: unknown node shape: note
            # AttributeError: Unknown attribute: NodeGroup.background
            # RuntimeError: could not belong to two groups: processor
            raise InvalidDataSentError("Can't process input. Details: %s" % str(e))
        except Exception:
            logger.exception('Formatter raised unhandled exception')
            raise RenderingFailedError('Formatter raised unhandled exception')

        if len(stderr) == 0 and len(result) == 0:
            raise InvalidDataSentError("Can't process input")  # special case for blockdiag/seqdiag

        if data['output'] == 'png' and 'width' in data and 'height' in data:
            resized_image_stream = formatter.resize(BytesIO(result), data['width'], data['height'])
            result = resized_image_stream.read()

        return Response(
            {'image_source': b64encode(result) if data['output'] == 'png' and isinstance(result, bytes) else result}
        )
