import logging
import time
from uuid import uuid4

from django.core.cache import caches
from rest_framework.serializers import Serializer

from wiki.api_core.framework import Response
from wiki.api_core.framework import WikiAPIView
from wiki.api_core.raises import raises
from wiki.api_core.waffle_switches import MDB_REDIS_VIEWS

logger = logging.getLogger(__name__)


class RedisDiagSetView(WikiAPIView):
    FEATURE_FLAG = MDB_REDIS_VIEWS

    serializer_class = Serializer

    @raises()
    def post(self, request, *args, **kwargs):
        key = request.data['key']
        value = request.data['value']

        cache = caches['redis_test']

        elapsed_time = time.time()
        cache.set(key, value)
        elapsed_time = time.time() - elapsed_time

        return Response({'elapsed_time': elapsed_time})


class RedisDiagGetView(WikiAPIView):
    FEATURE_FLAG = MDB_REDIS_VIEWS

    serializer_class = Serializer

    @raises()
    def post(self, request, *args, **kwargs):
        logger.info('getting key')
        key = request.data['key']

        logger.info('getting redis_test cache')
        cache = caches['redis_test']

        elapsed_time = time.time()

        logger.info('getting cache value')
        cache.get(key)
        elapsed_time = time.time() - elapsed_time

        return Response({'elapsed_time': elapsed_time})


class RedisDiagLockView(WikiAPIView):
    FEATURE_FLAG = MDB_REDIS_VIEWS

    serializer_class = Serializer

    @raises()
    def post(self, request, *args, **kwargs):
        cache = caches['redis_test']

        lock_sum = 0
        unlock_sum = 0
        times = 1000
        for i in range(times):
            lock = cache.lock('random_key', blocking_timeout=1)
            t_0 = time.time()
            lock.__enter__()
            lock_sum += time.time() - t_0
            t_0 = time.time()
            lock.__exit__(None, None, None)
            unlock_sum += time.time() - t_0
        lock_sum /= times
        unlock_sum /= times

        return Response({'lock_sum': lock_sum, 'unlock_sum': unlock_sum})


class RedisDiagMakeLockView(WikiAPIView):
    FEATURE_FLAG = MDB_REDIS_VIEWS

    serializer_class = Serializer

    @raises()
    def post(self, request, *args, **kwargs):
        cache = caches['redis_test']
        uuid = str(uuid4())

        logger.info(f'{uuid}: before lock')
        elapsed = time.time()

        try:
            with cache.lock('masstermax', blocking_timeout=0):
                logger.info(f'{uuid}: inside the lock')
                time.sleep(5)
            logger.info(f'{uuid}: after lock')
            locked = False
        except Exception as e:
            logger.info(f'{uuid}: error in lock was: {e}')
            locked = True

        return Response({'was_locked': locked, 'elapsed_time': time.time() - elapsed})
