from django.conf import settings
from django.http import HttpResponseBadRequest
from rest_framework.response import Response

from wiki.api_core.raises import raises
from wiki.api_svc.logic.suggest import user_layer, group_layer, department_layer
from wiki.api_svc.serializers import SuggestRequest
from wiki.api_svc.views.base import ServiceAPIView

layer_handlers = {
    'people': user_layer,
    'groups': group_layer,
    'departments': department_layer,
}


class SuggestView(ServiceAPIView):
    serializer_class = SuggestRequest

    """
    Саджест для b2b, возвращает список пользователей, групп и/или департаментов
    имеющих в своем названии/логине переданный текст, пример ответа:
    [
        {
            "layer": "people",
            "result": [
               "title": "Name",
                "url": "",
                "id": "113000001299453",
                "cloud_id": "",
                "fields": [
                    {
                        "type": "login",
                        "value": "abracadabra"
                    },
                    {
                        "type": "department_name",
                        "value": "Все сотрудники"
                    },
                    {
                        "type": "is_dismissed",
                        "value": false
                    },
                    {
                        "type": "avatar_url",
                        "value": "https://avatars.mds.yandex.net/get-yapic/islands-middle"
                    }
                ],
            ]
        },
        {
            "layer": "departments",
            "result": [
                {
                    "id": "1",
                    "url": "https://connect.yandex.ru/staff/departments/1",
                    "title": "Все сотрудники",
                    "fields": [],
                    "layer": "departments",
                    "click_urls": []
                }
            ],
        },
        {
            "layer": "groups",
            "result": [
                {
                    "id": "5",
                    "url": "https://connect.yandex.ru/staff/groups/5",
                    "title": "PartyGroup",
                    "fields": [],
                    "layer": "groups",
                    "click_urls": []
                }
            ],
        }

    ]
    """

    @raises()
    def get(self, request, *args, **kwargs):
        if not settings.IS_BUSINESS:
            return HttpResponseBadRequest(content='Wrong environment, only for business')

        serializer = self.get_serializer(data=request.GET)
        serializer.is_valid(raise_exception=True)
        text = serializer.validated_data['text']
        layers = serializer.validated_data['layers']

        if request.GET.get('version') == '2':
            return self.version_2(layers, text)

        return self.version_default(layers, text)

    def version_default(self, layers, text):
        result = []
        for layer in layers.split(','):
            if layer in layer_handlers:
                handler = layer_handlers[layer]
                result.append({'layer': layer, 'result': handler(text, '1')})

        return Response(result)

    def version_2(self, layers, text):
        out = {}
        for layer in layers.split(','):
            if layer in layer_handlers:
                handler = layer_handlers[layer]
                out[layer] = {
                    'result': handler(text, '2'),
                    'pagination': {'page': 0, 'per_page': 0, 'pages': 0, 'count': 0},  # тыква
                }
        return Response(out)
