
from django.conf import settings

from wiki.org import get_org_id
from wiki.utils.mongo import get_mongo_client
from wiki.utils.timezone import now


def _get_page_catalog():
    """
    Возвращает коллекцию page_catalog.
    Строит индексы, если индексы еще не были построены.
    """
    db = get_mongo_client()
    db.page_catalog.create_index('modified_at')
    db.page_catalog.create_index('pk')
    db.page_catalog.create_index('org_id')
    return db.page_catalog


def _get_page_catalog_meta():
    """
    Возвращает коллекцию page_catalog_meta.
    Строит индексы, если индексы еще не были построены.
    """
    db = get_mongo_client()
    db.page_catalog_meta.create_index('org_id')
    return db.page_catalog_meta


def get_and_store_last_update_started_at():
    """
    Возвращает дату начала предыдущего обновления каталога страниц,
    и запоминает текущую дату для следующего обновления.

    Если каталог обновляется впервые, либо дату начала предыдущего обновления
    сознательно удалили (например, для форсирования обновления всех страниц),
    то возвращается None.

    Дата начала предыдущего обновления хранится в коллекции page_catalog_meta,
    содержащую документы вида:

    {
      "org_id": <id>,
      "last_update_started_at": <data>
    },
    ...

    """
    meta = _get_page_catalog_meta()

    org_id = get_org_id()

    doc = meta.find_one({'org_id': org_id})

    last_update_started_at = doc['last_update_started_at'] if doc else None

    meta.replace_one(
        {'org_id': org_id},
        {'org_id': org_id, 'last_update_started_at': now()},
        upsert=True,
    )

    return last_update_started_at


def remove_store_last_update_started_at():
    """
    Удалить дату начала предыдущего обновления каталога страниц.
    """
    _get_page_catalog_meta().remove({'org_id': get_org_id()})


def to_page_catalog_format(page, users=None, groups=None, users_ext=None, groups_ext=None):
    """
    Привести данные о странице в формат, хранящийся в MongoDB.

    @param page: вики страница
    @type page: wiki.pages.models.page.Page

    @param users: список пользователей, имеющих доступ к странице
    @type users: list

    @param groups: список групп, имеющих доступ к странице
    @type groups: list

    @param users_ext: список внешних пользователей, имеющих доступ к внешней странице
    @type users_ext: list

    @param groups_ext: список внешних групп, имеющих доступ к внешней странице
    @type groups_ext: list

    @return: данные страницы для вставки в MongoDB
    @rtype: dict
    """

    def get_users_uid(users_list):
        return [u.uid for u in users_list] if users_list else None

    def get_groups_url(groups_list):
        return [g.url for g in groups_list] if groups_list else None

    org_id = get_org_id()

    data = {
        'include_in_search': not page.excluded_from_search_index,
        'uri': page.supertag,
        'deleted': page.status == 0,
        'modified_at': page.modified_at_for_index,
        'pk': page.pk,
        'org_id': org_id,
    }

    if not settings.IS_BUSINESS:
        # Эти данные нужны только для Интранета.
        data.update(
            {
                'title': page.title,
                'users': get_users_uid(users),
                'users_ext': get_users_uid(users_ext),
                'groups': get_groups_url(groups),
                'groups_ext': get_groups_url(groups_ext),
                'opened_to_external': page.opened_to_external_flag,
            }
        )

    return data


def insert_pages(pages):
    """
    Сохранить данные о страницах в MongoDB.

    @param pages: сохраняемые страницы
    @type pages: list
    """
    _get_page_catalog().insert(pages)


def remove_all_pages():
    """
    Удалить все данные о страницах из MongoDB.
    """
    _get_page_catalog().remove({'org_id': get_org_id()})


def remove_pages_by_pk(pk_list):
    """
    Удалить данные о страницах с указанными pk из MongoDB.
    """
    _get_page_catalog().remove({'pk': {'$in': pk_list}})


# Поля страницы в базе, ненужные в ответе ручки.
# Поле pk, на самом деле, тоже ненужно в ручке,
# но его пришлось оставить, поскольку нужно вычислять
# параметр gt_pk для следующего запроса по последнему элементу выборки.
_exclude_page_fields = ('_id', 'deleted', 'modified_at', 'org_id')

_projection = {exclude_field: False for exclude_field in _exclude_page_fields}


def get_pages(since=None, limit=None, not_deleted=False, gt_pk=None, all_fields=False):
    """
    Получить данные о страницах из MongoDB.

    @param since: Получить только страницы с modified_at >= since.
    @type since: datetime

    @param limit: Ограничение размера выборки.
    @type limit: int

    @param not_deleted: Если равно True, то запрашиваем только неудаленные страницы.
    @type not_deleted: bool

    @param gt_pk: Начиная с какого primary key страницы делать выборку. pk: {$gt: <gt_pk>} работает быстрее, чем skip().
    @type gt_pk: int

    @param all_fields: Если равно True, то возвращать все поля из базы, включая ненужные из списка _exclude_page_fields.
    @type all_fields: bool

    @rtype list
    """
    query = {}
    org_id = get_org_id()

    if since:
        query['modified_at'] = {'$gte': since}
    if not_deleted:
        query['deleted'] = False
    if gt_pk:
        query['pk'] = {'$gt': gt_pk}
    if org_id:
        query['org_id'] = org_id

    projection = None if all_fields else _projection

    res = _get_page_catalog().find(query, projection)

    if limit:
        res.limit(limit)

    res.sort('pk', 1)

    return list(res)
