
import logging
import time
from datetime import datetime
from urllib.parse import urlencode

from wiki.api_v1.dao.page_catalog import (
    get_and_store_last_update_started_at,
    get_pages,
    insert_pages,
    remove_all_pages,
    remove_pages_by_pk,
    to_page_catalog_format,
)
from wiki.org import get_org
from wiki.pages.access import get_bulk_raw_access, interpret_raw_access
from wiki.pages.access.external import _interpret_access_for_all_pages, _interpret_external_access_for_all_pages
from wiki.pages.models import Page
from wiki.utils.lock import FailedToGetLockError, check_lock_acquired
from wiki.utils.models import get_chunked
from wiki.utils.timezone import make_aware_utc

logger = logging.getLogger(__name__)


def load_page_catalog(params, handler_url, is_searchbot=False):
    """
    Вернуть данные каталога страниц для заданных параметров.

    @param params: Параметры загрузки каталога страниц.
    @type params: PageCatalogQueryParams

    @param handler_url: Абсолютный URL запроса без параметров, нужен для составления next.
    @type handler_url: basestring

    @param is_searchbot: фильтруем страницы, которые не надо индексировать боту

    @rtype: dict
    """

    pages = []
    need_next_request = False
    next_gt_pk = params.gt_pk
    since = make_aware_utc(datetime.utcfromtimestamp(params.since)) if params.since else None
    status = 'ok'

    try:
        # Каталог страниц в MongoDB обновляется не транзакционно -
        # происходит несколько REMOVE/INSERT операций (см. update_page_catalog()).
        # Транзакционность обновления каталога обеспечивается за счет использования лока
        # 'page_catalog_lock' - его захватывает задача celery, обновляющая каталог, а данная ручка проверяет, что лок
        # не захвачен таской.
        if not check_lock_acquired('page_catalog_lock'):
            pages = get_pages(since=since, limit=params.limit, not_deleted=params.not_deleted, gt_pk=params.gt_pk)

        # Следующий запрос надо делать, если текущий был с limit, и размер загруженного чанка не меньше limit.
        if params.limit and len(pages) >= params.limit:
            next_gt_pk = pages[-1]['pk']
            need_next_request = True

        # Если это поисковый бот, отфильтруем страницы
        if is_searchbot:
            pages = [page for page in pages if page.get('include_in_search', True)]

    except FailedToGetLockError as e:
        logger.warn(str(e))
        need_next_request = True
        status = 'retry'

    next = (
        '{handler_url}?{query_params}'.format(
            handler_url=handler_url,
            query_params=PageCatalogQueryParams(
                since=params.since, limit=params.limit, not_deleted=params.not_deleted, gt_pk=next_gt_pk
            ).encode(),
        )
        if need_next_request
        else None
    )

    return {'timestamp': int(round(time.time())), 'status': status, 'next': next, 'pages': pages}


def update_page_catalog():
    """
    Обновить каталог страниц в MongoDB.
    """

    # Мы обновляем только те страницы, что изменились с
    # даты начала предыдущего обновления. Если каталог страниц
    # обновляется впервые, либо дату начала предыдущего обновления
    # сознательно удалили, то обновляем все страницы.
    last_update_started_at = get_and_store_last_update_started_at()

    if not last_update_started_at:
        # На случай сознательного удаления даты начала предыдущего обновления.
        remove_all_pages()

    org = get_org()

    pages_qs = Page.objects.filter(org=org).select_related('searchexclude')
    if last_update_started_at:
        pages_qs = pages_qs.filter(modified_at_for_index__gte=last_update_started_at)

    updated_pages_count = 0

    for page_chunk in get_chunked(pages_qs):

        raw_access_list = get_bulk_raw_access(page_chunk)
        page_data_chunk = []

        for page, raw_access in raw_access_list.items():

            interpreted_access = interpret_raw_access(raw_access)
            page, users, groups = _interpret_access_for_all_pages(page, interpreted_access)

            users_ext = None
            groups_ext = None
            if page.opened_to_external_flag:
                _, users_ext, groups_ext = _interpret_external_access_for_all_pages(page, interpreted_access)

            page_data = to_page_catalog_format(page, users, groups, users_ext, groups_ext)
            page_data_chunk.append(page_data)

        if last_update_started_at:
            # Перед обновлением данных о страницах удаляем старые данные.
            remove_pages_by_pk([data['pk'] for data in page_data_chunk])

        insert_pages(page_data_chunk)

        updated_pages_count += len(page_data_chunk)

    logger.info(
        '%d pages are updated in page catalog %s',
        updated_pages_count,
        '[modified_at > %s]' % last_update_started_at if last_update_started_at else '[full update]',
    )


class PageCatalogQueryParams:
    """
    Параметры запроса к каталогу страниц.
    """

    def __init__(self, since=None, limit=None, not_deleted=False, gt_pk=None):
        """
        @type since: int
        @type limit: int
        @type not_deleted: bool
        @type gt_pk: int
        """
        self.since = since
        self.limit = limit
        self.not_deleted = not_deleted
        self.gt_pk = gt_pk

    def encode(self):
        """
        Кодировать параметры в URL.

        @rtype basestring
        """
        params = {}
        if self.since:
            params['since'] = self.since
        if self.limit:
            params['limit'] = self.limit
        if self.not_deleted:
            params['not_deleted'] = 'true'
        if self.gt_pk:
            params['gt_pk'] = self.gt_pk
        return urlencode(params)

    @staticmethod
    def from_request_params(request_params):
        """
        @type request_params: dict
        @rtype PageCatalogQueryParams
        """
        return PageCatalogQueryParams(
            since=request_params['since'],
            limit=request_params['limit'],
            not_deleted=request_params['not_deleted'],
            gt_pk=request_params['gt_pk'],
        )
