
from rest_framework.response import Response

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.logic.files import FileUploadError, upload_file_to_storage
from wiki.api_core.raises import raises
from wiki.api_frontend.views.files import FileView as OriginalFileView
from wiki.api_v1.views.serializers.files import AttachedFile, ChangeFileSerializer, FileProxy


class FileView(OriginalFileView):
    """
    Работа с одним загруженным файлом.
    """

    serializer_class = ChangeFileSerializer
    available_content_types = ('application/json', 'application/x-www-form-urlencoded', 'multipart/form-data')
    check_readonly_mode = True

    @raises(InvalidDataSentError)
    def post(self, request, filename, *args, **kwargs):
        """
        Позволяет изменить загруженный в Вики файл.

        %%
        $curl -i -XPOST -F name=test -F filedata=@fabfile.py \
        https://wiki.yandex-team.ru:8080/_api/v1/pages/users/.files/filename.jpg/ \
        --data 'тело запроса'
        %%

        Пример тела запроса
        %%
        {
          "description": "Greenday!"
        }
        %%

        Вы заливаете новый файл на место старого? Ручка позволит вам поменять тип
        контента залитого файла. Если у файла был тип "image/png", то ручка позволит
        залить только "application/js", например. Учтите, что так заливать не надо.
        Контент, который был внедрен на вики и отображен в виде HTML (например,
        с помощью тега IMG), можно подменить каким-нибудь javascript-файлом. И что-нибудь
        сломать. Это место остается за потребителем.

        Возвращает json c атрибутами файла вида

        %%(js)
        {"data": {
             "description": "Another text file",
             "docviewer_url": "http://docviewer.dst.yandex-team.ru/?url=ya-wiki%3A//wiki-unstable.dev.yandex-team.ru/filespages/myfilename-1.doc",
             "name": "myfilename.doc",
             "size": 0.01,
             "upload_date": "2014-08-12T17:14:15",
             "url": "/filespages/.files/myfilename-1.doc",
             "user_name": "thasonic"
        }}
        %%
        """  # noqa
        serializer = self.get_serializer(data=request.data)
        if not serializer.is_valid():
            raise InvalidDataSentError(serializer.errors)

        if list(request.FILES.values()):
            try:
                storage_id = upload_file_to_storage(request)
            except FileUploadError as exc:
                raise InvalidDataSentError(str(exc))
        else:
            storage_id = None

        file = self.get_file_by_filename(filename, request.page)
        if file is None:
            raise InvalidDataSentError('File with filename "{0}" does not exist'.format(filename))

        file = serializer.save(file, storage_id)

        return Response(AttachedFile(FileProxy(file)).data)
