
from rest_framework.response import Response

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.framework import WikiAPIView
from wiki.api_core.logic import files as files_logic
from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.io import EmptySerializer
from wiki.api_frontend.views.files import FileUploadView as FrontendFileUploadView


class FileUploadView(WikiAPIView):
    """
    Загрузка файлов.
    """

    available_content_types = FrontendFileUploadView.available_content_types
    serializer_class = EmptySerializer

    @raises(InvalidDataSentError)
    def post(self, *args, **kwargs):
        """
        Загружает в хранилище первый файл из запроса и возвращает ключ этого файла в хранилище.
        Файлы в запросе должны быть такие же, как посылает html форма -
        запрос с content-type: multipart/form-data;

        %%
        $curl -i -F name=test -F filedata=@fabfile.py https://wiki.yandex-team.ru:8080/_api/v1/files/
        %%

        Возвращает json c storage_id файла вида
        {"data": {
            "storage_id": "221/wiki:file:123:12946:2014-11-18 10:47:28:534026"
        }}
        """
        try:
            storage_id = files_logic.upload_file_to_storage(self.request)
        except files_logic.FileUploadError as exc:
            raise InvalidDataSentError(str(exc))
        return Response(
            {
                'storage_id': storage_id,
            }
        )
