
import logging

from django.conf import settings
from django.db import transaction
from django.urls import reverse
from rest_framework.response import Response

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.errors.permissions import UserHasNoAccess
from wiki.api_core.framework import PageAPIView
from wiki.api_core.logic import files as upload_logic
from wiki.api_core.pagination_utils import StreamPaginator
from wiki.api_core.raises import raises
from wiki.api_v1.views.serializers.files import (
    AttachedFile,
    AttachFilesResponseSerializer,
    FileProxy,
    FilesAttachSerializer,
)
from wiki.files.models import File
from wiki.pages.access import is_admin

logger = logging.getLogger(__name__)


def validate_access_to_file(request):
    """
    Бросить UserHasNoAccess, если нет доступа к файлу.
    """
    if request.from_yandex_server:
        raise UserHasNoAccess('YaServer has no access')

    user = request.user
    if not (is_admin(user) or request.page.has_access(user)):
        raise UserHasNoAccess


def get_files(page_id, limit, start_id):
    """
    Отпагинировать список файлов, прикрепленных к странице.
    """
    files = File.active.filter(page_id=page_id).select_related(
        'user',  # потому что FileProxy нужно это поле
    )
    return StreamPaginator(files).paginate(limit, start_id)


class FilesView(PageAPIView):
    """
    View для прикрепления к странице файлов, загруженных в хранилище
    через ручку %%/v1/files/%%.
    """

    serializer_class = FilesAttachSerializer
    PARAM_PER_PAGE = 'per_page'
    PARAM_START_ID = 'start_id'
    available_content_types = ('application/json', 'application/x-www-form-urlencoded', 'multipart/form-data')

    def check_page_access(self):
        """
        Проверить, что у пользователя есть права на запись.
        """
        validate_access_to_file(self.request)

    @raises()
    def get(self, request, *args, **kwargs):
        """
        Вернуть список файлов, прикрепленных к странице.

        Принимает GET-параметры per_page и start_id для пагинации.

        Пример запроса:

        %%(sh)
        $ curl -H "Authorization: OAuth <token>" \
        "https://wiki-api.yandex-team.ru/_api/v1/pages/<supertag>/.files?per_page=25&start_id=548"
        %%

        Параметр per_page игнорирует значения больше 2500.

        Пример ответа:

        %%(js)
        {
            "data": {
                "files": [
                    {
                        "upload_date": "2014-04-10 12:10:33",
                        "description": "описалово",
                        "url": "/wiki/.files/rabotasgit.pdf",
                        "docviewer_url": "http://docviewer.dst.yandex-team.ru/?url=",
                        "size": "0.13",
                        "user_name": "user3371",
                        "name": "readme.txt"
                    },
                    {
                        "upload_date": "2014-04-10 12:10:33",
                        "description": "Мой текстовый файл.JPG",
                        "url": "/wiki/.files/rea.txt",
                        "docviewer_url": "http://docviewer.dst.yandex-team.ru/?url=",
                        "size": "4.31",
                        "user_name": "user3371",
                        "name": "rea.txt"
                    },
                ],
                "links": {
                    "next": "https://wiki-api.yandex-team.ru/_api/v1/hanna/banana/.files?per_page=25&start_id=3020"
                }
            },
            "user": {
                ....
            }
            "debug": {
                ....
            },
        }
        %%
        """
        start_id = request.GET.get(self.PARAM_START_ID, None)
        limit = min(int(request.GET.get(self.PARAM_PER_PAGE, 25)), 2500)
        stream = get_files(self.request.page.id, limit, start_id)
        next_link = self.next_link(stream.next_start_id, limit) if stream.next_start_id else None
        data = {'files': (FileProxy(file) for file in stream), 'links': None}
        if next_link:
            data['links'] = {'next': next_link}
        return Response(AttachFilesResponseSerializer(data).data)

    def next_link(self, next_id, per_page):
        path = reverse(
            'api_v1:pages:file_collection',
            kwargs={
                'tag': self.request.page.supertag,
            },
        )
        url = '{protocol}://{hostname}{path}?{per_page_param}={per_page}'.format(
            protocol=settings.WIKI_PROTOCOL,
            hostname=settings.NGINX_HOST,
            path=path,
            per_page_param=self.PARAM_PER_PAGE,
            per_page=per_page,
        )
        if next_id is not None:
            url += '&start_id=' + str(next_id)
        return url

    @transaction.atomic()
    @raises()
    def post(self, request, *args, **kwargs):
        """
        Добавить на страницу файлы, загруженные ранее в хранилище.
        Вернуть список файлов, прикрепленных к странице текущим запросом.
        Формат ответа полностью совпадает с форматом ручки %%/.files%%.

        Пример запроса:

        %%(http)
        POST /_api/v1/pages/users/elisei/test/.files
        %%

        Тело запроса:

        %%(js)
        {
            // список id файлов (storage_id, полученные из ручки /.upload)
            "files": [{
                "storage_id": "wiki:file:readme.txt:126845:2014-04-09 20:51:46:593471",
                "description: "описалово"
            }, {
                "storage_id": "wiki:file:rea.txt:6845:2014-04-09 20:51:46:593471",
                "description: "Мой текстовый файл.JPG"
            }]
        }
        %%

        Пример ответа:

        %%(js)
        {
            "debug": {
                ....
            },
            "data": {
                "files": [
                    {
                        "upload_date": "2014-04-10 12:10:33",
                        "description": "описалово",
                        "url": "/wiki/.files/rabotasgit.pdf",
                        "docviewer_url": "http://docviewer.dst.yandex-team.ru/?url=",
                        "size": "0.13",
                        "user_name": "user3371",
                        "name": "readme.txt"
                    },
                    {
                        "upload_date": "2014-04-10 12:10:33",
                        "description": "Мой текстовый файл.JPG",
                        "url": "/wiki/.files/rea.txt",
                        "docviewer_url": "http://docviewer.dst.yandex-team.ru/?url=",
                        "size": "4.31",
                        "user_name": "user3371",
                        "name": "rea.txt"
                    },
                ]
            },
            "user": {
                .... // пользователь, сделавший запрос
            }
        }
        %%

        Поле "description" ограничено 255 символами.
        """
        serializer = self.get_serializer(data=request.data)
        if serializer.is_valid():
            files = serializer.save(self.request.page, self.request.user)
            return Response(
                AttachFilesResponseSerializer(
                    {
                        'files': (FileProxy(file) for file in files),
                        'links': None,
                    }
                ).data
            )
        else:
            raise InvalidDataSentError(serializer.errors)

    @raises()
    def put(self, request, *args, **kwargs):
        """
        Загрузить файл, прикрепленный к странице.

        %%(sh)
        $ curl -H "Authorization: OAuth <token>" -XPUT \
        "https://wiki-api.yandex-team.ru/_api/v1/<supertag>/.files?per_page=25&start_id=548"
        --data 'тело запроса'
        %%

        Тело запроса
        %%(js)
        {
            "description": "Another text file",
        }
        %%

        Пример ответа:

        %%(js)
        {
            "description": "Another text file",
             "docviewer_url": "http://docviewer.dst.yandex-team.ru/?url=ya-wiki%3A//wiki.dev.y-t.ru/files/filename.doc",
             "name": "myfilename.doc",
             "size": 0.01,
             "upload_date": "2014-08-12T17:14:15",
             "url": "/filespages/.files/myfilename-1.doc",
             "user_name": "thasonic"
        }
        %%
        """
        try:
            storage_id = upload_logic.upload_file_to_storage(request)
        except upload_logic.FileUploadError as exc:
            raise InvalidDataSentError(str(exc))
        data = {
            'files': [
                {
                    'storage_id': storage_id,
                    'description': request.data.get('description', ''),
                }
            ]
        }
        serializer = self.get_serializer(data=data)
        if serializer.is_valid():
            file = serializer.save(self.request.page, self.request.user)[0]
            return Response(AttachedFile(FileProxy(file)).data)
        else:
            raise InvalidDataSentError(serializer.errors)
