
from django.conf import settings
from rest_framework.response import Response

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.errors.permissions import ExternalAccessDenied
from wiki.api_core.framework import WikiAPIView
from wiki.api_core.raises import raises
from wiki.api_core.utils import DirectoryRobotPermission, TVMIDPermission, UIDListPermission, is_tvm_authentication
from wiki.api_v1.logic.page_catalog import PageCatalogQueryParams, load_page_catalog
from wiki.api_v1.views.serializers.page_catalog import PageCatalogSerializer


class IntranetPermission(UIDListPermission):
    uid_list = settings.PAGE_CATALOG_ACCESS_UIDS


class B2BPermission(DirectoryRobotPermission):
    service_slug = 'search'


# TODO удалить, когда поиск и СИБ полностью перейдут на TVM2
Permission = B2BPermission if settings.IS_BUSINESS else IntranetPermission


class TVMPermission(TVMIDPermission):
    tvm_client_ids = settings.PAGE_CATALOG_ACCESS_TVM_CLIENT_IDS
    strict = False  # Пока в эту ручку ходят сервисы по OAuth не отрубаем запросы которые авторизованы не по TVM2


class PageCatalogView(WikiAPIView):
    """
    View для доступа к каталогу страниц.

    см. https://wiki.yandex-team.ru/wiki/api/page-catalog/

    """

    permission_classes = (TVMPermission, Permission)

    @raises()
    def get(self, request, *args, **kwargs):
        if not settings.IS_INTRANET and not settings.IS_BUSINESS:
            raise ExternalAccessDenied

        serializer = PageCatalogSerializer(data=request.GET)

        if not serializer.is_valid():
            raise InvalidDataSentError(serializer.errors)

        params = PageCatalogQueryParams.from_request_params(serializer.validated_data)

        handler_url = request.build_absolute_uri().split('?')[0]

        is_searchbot = (
            is_tvm_authentication(request) and str(request.yauser.service_ticket.src) == settings.SEARCH_TVM2_CLIENT_ID
        )

        return Response(load_page_catalog(params, handler_url, is_searchbot))
