
from django.conf import settings

from wiki.api_core.framework import HttpResponseRedirect, PageAPIView, Response
from wiki.api_core.raises import raises
from wiki.api_core.utils import DirectoryRobotPermission, TVMIDPermission, UIDListPermission, is_tvm_authentication
from wiki.api_v1.views.serializers.page_search_index import PageSearchIndexSerializer
from wiki.utils.wiki_robot import get_wiki_robot
from wiki.utils.request import ratelimiter


RPS_LIMIT = int(settings.PAGE_SEARCH_INDEX_RPS)


class IntranetPermission(UIDListPermission):
    uid_list = settings.PAGE_SEARCH_INDEX_ACCESS_UIDS


class B2BPermission(DirectoryRobotPermission):
    service_slug = 'search'


# TODO удалить, когда поиск полностью перейдет на TVM2
Permission = B2BPermission if settings.IS_BUSINESS else IntranetPermission


class TVMPermission(TVMIDPermission):
    tvm_client_ids = settings.PAGE_SEARCH_INDEX_ACCESS_TVM_CLIENT_IDS
    strict = False  # Пока в эту ручку ходят сервисы по OAuth не отрубаем запросы которые авторизованы не по TVM2


class PageSearchIndexView(PageAPIView):
    """
    View, отдающее данные о страницах и гридах поиску в виде json.

    https://wiki.yandex-team.ru/wiki/components/wiki-dlja-poiska/#ruchkastranicy
    """

    permission_classes = (TVMPermission, Permission)

    @ratelimiter(rate=RPS_LIMIT)
    @raises()
    def get(self, request, *args, **kwargs):
        """
        Получить данные о странице или гриде.

        %%
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/v1/pages/<supertag>/.search-index?lang=en"
        %%

        """
        page = self.request.page
        if page.has_redirect():
            return HttpResponseRedirect('/%s' % page.redirects_to.supertag)

        return Response(PageSearchIndexSerializer(page, context={'request': self.request}).data)

    def check_page_access(self):
        # Поисковый робот должен получать доступ к данным страницы
        # вне зависимости от доступов страницы.
        if is_tvm_authentication(self.request):
            # Если Поиск пришел только с сервисным тикетом, т.е. обеличенно, то все динамические экшены выполняем
            # от имени нашего робота
            self.request.user = get_wiki_robot()
