"""
Сериализаторы файлов
"""
import logging
from datetime import timedelta
from urllib.parse import urlparse

from django.conf import settings
from rest_framework import serializers

from wiki.api_core.serializers import LinksSerializer
from wiki.api_core.utils import is_docviewer_previewable
from wiki.files.models import MDS_STORAGE, File, fileinfo_by_storage_id
from wiki.notifications.models import EventTypes, PageEvent
from wiki.sync.connect.org_ctx import get_org_dir_id
from wiki.utils import timezone
from wiki.utils.docviewer import docviewer_preview_link

logger = logging.getLogger(__name__)


def now():
    return timezone.now()


class FileAttachSerializer(serializers.Serializer):
    storage_id = serializers.CharField()
    description = serializers.CharField(required=False, allow_blank=True)


class FilesAttachSerializer(serializers.Serializer):
    files = serializers.ListField(child=FileAttachSerializer(required=True))

    def validate(self, attrs):
        attrs = super(FilesAttachSerializer, self).validate(attrs)
        for file_obj in attrs['files']:
            storage_id = file_obj['storage_id']
            if not MDS_STORAGE.exists(storage_id):
                logger.error('No such file with id "%s" in storage', storage_id)
                # так как вызовы ПОСЛЕДОВАТЕЛЬНЫ, то для ускорения работы
                # бросаем ошибку сразу, не проверив оставшиеся storage_ids
                raise serializers.ValidationError('No storage_id "%s" in storage' % storage_id)
        return attrs

    def save(self, page, owner, **kwargs):
        """
        @rtype: list
        """
        saved_files = []
        NOW = now()  # чтобы все файлы датировались одним моментом
        for file_obj in self.data['files']:
            saved_files.append(
                save_file(
                    now=NOW,
                    page=page,
                    owner=owner,
                    storage_id=file_obj['storage_id'],
                    description=file_obj.get('description', ''),
                )
            )
        return saved_files


def save_file(now, page, owner, storage_id, description=''):
    """
    Сохранить информацию о прикрепленном файле в базе.

    @type now: datetime
    @type storage_id: unicode
    @rtype: File
    """
    file_name, file_size = fileinfo_by_storage_id(storage_id)

    attached_file = File(
        page=page,
        user=owner,
        name=file_name,
        url=File.get_unique_url(page, file_name),
        size=file_size,
        # поле description на клиенте в новом дизайне не предусмотрено
        description=description,
        mds_storage_id=storage_id,
    )

    attached_file.save()
    attached_file.created_at = now
    attached_file.modified_at = now
    attached_file.save()

    page.files += 1
    page.save()

    page_event = PageEvent(
        timeout=now + timedelta(minutes=20), page=page, author=owner, event_type=EventTypes.add_file, notify=True
    )
    page_event.meta = {
        'filename': attached_file.name,
        'url': attached_file.url,
        'size': attached_file.size,
        'id': attached_file.id,
    }
    page_event.save()

    return attached_file


class FileProxy(object):
    file = None

    def __init__(self, file):
        """
        @type file: File
        """
        self.file = file

    @property
    def url(self):
        full_path = self.file.full_path(self.file.page.url, self.file.url)
        return '{0}://{1}{2}'.format(settings.WIKI_PROTOCOL, settings.NGINX_HOST, full_path)

    @property
    def docviewer_url(self):
        if not is_docviewer_previewable(self.file):
            return None

        dir_org_id = get_org_dir_id()
        link = docviewer_preview_link(self.file, dir_org_id)

        parsed_link = urlparse(link)
        if parsed_link.hostname:
            return link
        # если ссылка не содержит хостнейма, то считаем, что эта ссылка ведет на вики
        return parsed_link._replace(netloc=settings.NGINX_HOST, scheme=settings.WIKI_PROTOCOL).geturl()

    @property
    def upload_date(self):
        return self.file.created_at

    @property
    def user_name(self):
        return self.file.user.username

    @property
    def size(self):
        return round(float(self.file.size) / 1024.0 / 1024, 2)  # in MB

    def __getattr__(self, item):
        return getattr(self.file, item)


class AttachedFile(serializers.Serializer):
    name = serializers.CharField()
    url = serializers.CharField()
    size = serializers.FloatField()
    description = serializers.CharField()
    user_name = serializers.CharField()
    docviewer_url = serializers.CharField(required=False)
    upload_date = serializers.DateTimeField(format='%Y-%m-%dT%H:%M:%S')


class AttachFilesResponseSerializer(serializers.Serializer):
    files = serializers.ListField(child=AttachedFile())
    links = LinksSerializer(allow_null=True)


class ChangeFileSerializer(serializers.Serializer):
    description = serializers.CharField(required=False, allow_blank=True)

    def save(self, file, storage_id, **kwargs):
        """
        Обновить файл.

        @type file: File
        @type storage_id: str
        @rtype: File
        """
        if self.data['description']:
            file.description = self.data['description']
        if storage_id:
            file.mds_storage_id = storage_id

        file.save()
        return file
