
import logging
from typing import Optional, Tuple

from rest_framework import serializers

from wiki.api_core.waffle_switches import INDEX_XLSX_PPTX, DISABLE_CLOUD_INDEXATION
from wiki.grids.models import Grid
from wiki.grids.utils import grid_raw_body
from wiki.integrations.ms.utils import get_doc_retriever_client
from wiki.pages.models import CloudPage, Page
from wiki.utils.context_bound_serializer import ContextBoundSerializer
from wiki.utils.dict_keypath import value_by_path
from wiki.utils.serializer_contexts import RequestContext
from wiki.utils.wfaas.client import get_wfaas_client

import waffle

logger = logging.getLogger(__name__)


class PageHtmlSerializer(ContextBoundSerializer[RequestContext], serializers.Serializer):
    def to_representation(self, page):
        error, html = self.load_html(page=page)
        if error:
            raise ValueError('Load html failed: %s' % error)

        data = {'html': html}

        return data

    def load_html(self, page):
        if page.page_type == Page.TYPES.CLOUD:
            return self._load_html_cloud_page(page)
        else:
            return self.raw_to_html(page)

    def _load_html_cloud_page(self, page: Page) -> Tuple[Optional[str], Optional[str]]:
        try:
            cloud_page = CloudPage.objects.get(page_id=page.id)
        except CloudPage.DoesNotExist:
            # пара битых страниц были созданы во время отладки.
            logger.error(f'No cloud page for page {page.supertag}')
            return None, ''

        if not cloud_page.cloud_src:
            return 'cloud page src is empty', None

        document_type = value_by_path('document.type', cloud_page.cloud_src, '')
        client = get_doc_retriever_client()

        methods = {
            'docx': client.docx_html,
        }

        if waffle.switch_is_active(INDEX_XLSX_PPTX):
            methods.update({'xlsx': client.xlsx_html, 'pptx': client.pptx_html})

        if waffle.switch_is_active(DISABLE_CLOUD_INDEXATION):
            methods = {}

        if document_type in methods:
            try:
                driveitem = cloud_page.cloud_src.get('driveitem')
                method = methods[document_type]
                response = method(driveitem)
            except Exception as e:
                return f'Request failed: {repr(e)}', None
        else:
            # Для неподдерживаемых возвращаем пустой html, иначе будем пятисотить когда за ними придет поиск
            return None, ''
        return None, response['html']

    def raw_to_html(self, page):
        page_body = ''

        wfaas_client = get_wfaas_client()
        to_html_method = wfaas_client.raw_to_html

        if page.page_type == Page.TYPES.GRID:
            grid = Grid.objects.get(pk=page.pk)
            page_body = grid_raw_body(grid)
        elif page.page_type == Page.TYPES.PAGE:
            page_body = page.body
        elif page.page_type == Page.TYPES.WYSIWYG:
            page_body = page.body
            to_html_method = wfaas_client.yfm_to_html

        try:
            html = to_html_method(page_body)
        except Exception as e:
            return f'wfaas client error: {repr(e)}', None

        return None, html
