import logging

from jinja2.filters import do_lower
from rest_framework import serializers

from wiki.api_v1.views.serializers.page_html import PageHtmlSerializer
from wiki.utils.context_bound_serializer import ContextBoundSerializer
from wiki.utils.serializer_contexts import RequestContext
from wiki.utils.supertag import tag_to_supertag
from wiki.utils.yandex_server_context import yaserver_context

logger = logging.getLogger(__name__)


class PageSearchIndexSerializer(ContextBoundSerializer[RequestContext], serializers.Serializer):
    def to_representation(self, page):
        data = self.load_yaserver_context(page)
        data.update(PageHtmlSerializer(page, context={'request': self.get_context().request}).data)

        return data

    def load_yaserver_context(self, page):
        ys_ctx = yaserver_context(page)
        breadcrumbs = page.breadcrumbs

        def parent_json(parent):
            supertag, title = parent
            res = {'supertag': supertag}
            if title:
                res['title'] = title
            return res

        cluster_one = breadcrumbs[0]

        ctx = {
            'title': page.title,
            'doc_group': 'wiki_%d' % page.id,
            'cluster_one': do_lower(cluster_one['name']),
            'cluster_one_name': cluster_one['name'],
            'cluster_one_supertag': tag_to_supertag(cluster_one['tag']),
            'cluster_one_tag': cluster_one['tag'],
            'cluster_one_url': cluster_one['url'],
            'breadcrumbs': [{'name': breadcrumb['name'], 'url': breadcrumb['url']} for breadcrumb in breadcrumbs],
            'wiki_page_tag': page.tag,
            'wiki_page_supertag': page.supertag,
            'wiki_page_url': page.url,
            'public': ys_ctx['is_public'],
            'usercluster': ys_ctx['is_user_cluster'],
            'ctime': ys_ctx['creation_time'],
            'mtime': ys_ctx['modification_time'],
            'owner_login': ys_ctx['owner_login'],
            'owner_full_name': ys_ctx['owner_full_name'],
            'owner_full_name_en': ys_ctx['owner_full_name_en'],
            'authors': ys_ctx['authors'],
            'linked_from_count': ys_ctx['linked_from_count'],
            'descendants_count': ys_ctx['descendants_count'],
            'favorited_count': ys_ctx['favorited_count'],
            'modifiers_count': ys_ctx['modifiers_count'],
            'files_count': page.files,
            'comments_count': page.comments,
            'is_documentation': ys_ctx['is_documentation'],
            'frontend_host': ys_ctx['frontend_host'],
            'parents': [parent_json(parent) for parent in ys_ctx['parents']],
            'editors': [
                {'login': editor.login, 'last_edit': editor.last_edition_date, 'count': editor.edited_times}
                for editor in ys_ctx['editors']
            ],
        }

        if 'is_official' in ys_ctx:
            ctx['is_official'] = ys_ctx['is_official']
        # TODO: WIKI-9069 - из метода yaserver_context() списки acl возвращаются как строки,
        # где значения соединены пробелами, но тут нам удобнее возвращать списки.
        # После выпиливания старой ручки для поиска сделать так, чтобы
        # yaserver_context() возвращал сразу списки.
        if ys_ctx['acl_groups_whitelist']:
            ctx['acl_groups_whitelist'] = ys_ctx['acl_groups_whitelist']
        if ys_ctx['acl_users_whitelist']:
            ctx['acl_users_whitelist'] = ys_ctx['acl_users_whitelist']
        if page.keywords_list:
            ctx['keywords'] = page.keywords_list
        if 'not_actual_since' in ys_ctx:
            ctx['not_actual_since'] = ys_ctx['not_actual_since']
            ctx['actual_pages'] = ys_ctx['actual_pages']
            ctx['actual_urls'] = ys_ctx['actual_urls']

        return ctx
