from functools import wraps
from typing import List

from django.conf import settings

from wiki.api_core.utils import is_tvm_authentication, is_authenticated
from wiki.api_v2.exceptions import Tvm2Only, NoOrganization, Forbidden


def private_auth(request):
    return is_tvm_authentication(request)


def public_auth(request):
    if settings.IS_BUSINESS:
        if not getattr(request, 'org', None):
            raise NoOrganization()

    return is_authenticated(request) and not request.user.is_anonymous


def support_auth(request):
    if request.user.is_anonymous:
        raise Forbidden()
    if not request.user.groups.filter(name=settings.IDM_ROLE_SUPPORT_GROUP_NAME).exists():
        raise Forbidden(debug_message='Only for IDM group "Support"')

    return is_authenticated(request) and not request.user.is_anonymous


def tvm2_only(client_ids: List[str] = None):
    def _tvm2_only(f):
        @wraps(f)
        def inner(*args, **kwargs):
            request = args[0]

            if not is_tvm_authentication(request):
                raise Tvm2Only()

            if client_ids is not None and request.yauser.service_ticket.src in client_ids:  # ???
                return f(*args, **kwargs)

            raise Tvm2Only()

        return inner

    return _tvm2_only
