import functools
import logging
import time
from collections import defaultdict

from django.http import Http404
from ninja import NinjaAPI
from ninja.errors import ValidationError as NinjaValidationError
from ylog.context import log_context

from wiki.api_core.errors.rest_api_error import RestApiError
from wiki.api_v2.exceptions import NinjaApiError, InternalServerError, ValidationError, NotFound
from wiki.utils.request_logging.context import extract_normalized_route

logger = logging.getLogger(__name__)


def upgrade_legacy_api_exceptions(fn):
    @functools.wraps(fn)
    def wrapper(*args, **kwargs):
        try:
            return fn(*args, **kwargs)
        except RestApiError as e:
            raise NinjaApiError.from_legacy_error(e)

    return wrapper


def wrap_log_context(fn):
    @functools.wraps(fn)
    def wrapper(request, *args, **kwargs):
        slug = getattr(request, 'supertag', None)
        route = extract_normalized_route(request)
        execution_time_ms = int((time.time() - request.start_time) * 1000)

        with log_context(slug=slug, route=route, execution_time_ms=execution_time_ms):
            return fn(request, *args, **kwargs)

    return wrapper


def subscribe_default_exception_handlers(api: NinjaAPI):
    @api.exception_handler(NinjaApiError)
    @wrap_log_context
    def ninja_api_error(request, exc: NinjaApiError):
        return api.create_response(
            request,
            exc.to_dict(),
            status=exc.status_code,
        )

    @api.exception_handler(RestApiError)
    @wrap_log_context
    def rest_api_error(request, exc: RestApiError):
        return api.create_response(
            request,
            exc.as_apiv2_dict(),
            status=exc.status_code,
        )

    @api.exception_handler(Exception)
    @wrap_log_context
    def any_error(request, exc):
        logger.exception(f'Unhandled exception in API: {exc} ({exc.__class__})')
        return ninja_api_error(request, InternalServerError(debug_message=f'{exc} ({exc.__class__})'))

    @api.exception_handler(NinjaValidationError)
    @wrap_log_context
    def validation_error(request, exc: NinjaValidationError):
        details = defaultdict(lambda: defaultdict(list))

        for error in exc.errors:
            # поле loc изначально идет из pydantic,
            # затем обогащается дополнительной информацией в django-ninja
            # и сюда приходит примерно в таком формате
            # ('location source', 'param name', ['field name', 'optional info']) например
            # ('query', 'page')
            # ('body', 'data', 'tags', 1) tags - имя поля, 1 - индекс элемента в массиве
            source = error['loc'][0]
            if len(error['loc']) > 2:
                field_name = error['loc'][2]
            else:
                field_name = error['loc'][1]

            error_msg = {'error_code': error['type'], 'debug_message': error['msg'], 'details': {}}

            if len(error['loc']) > 3:
                error_msg['details']['index'] = error['loc'][3]

            if context := error.get('ctx', None):
                error_msg['details']['context'] = context

            details[source][field_name].append(error_msg)

        return ninja_api_error(request, ValidationError(details=details))

    # @api.exception_handler(HttpResponseForbidden)
    # def forbidden_error(request, exc):
    #     return ninja_api_error(
    #         request, Forbidden()
    #     )
    #
    # @api.exception_handler(HttpResponseNotAllowed)
    # def not_allowed_error(request, exc):
    #     return ninja_api_error(
    #         request, NotAllowed()
    #     )
    #
    @api.exception_handler(Http404)
    @wrap_log_context
    def not_found_error(request, exc):
        return ninja_api_error(request, NotFound())
