from datetime import date
from enum import Enum
from typing import Union

from ninja import Schema
from pydantic import UUID4, constr

from wiki.api_v2.public.pages.page_identity import PageIdentity


CellValueTypes = Union[float, str, list[str], None]


class GridColumnTypes(str, Enum):
    STRING = 'string'
    NUMBER = 'number'
    DATE = 'date'
    SELECT = 'select'
    STAFF = 'staff'
    CHECKBOX = 'checkbox'
    TICKET = 'ticket'
    # ----
    TICKET_ASSIGNEE = 'ticket-assignee'
    TICKET_REPORTER = 'ticket-reporter'
    TICKET_STATUS = 'ticket-status'
    TICKET_PRIORITY = 'ticket-priority'
    TICKET_TYPE = 'ticket-type'
    TICKET_SUBJECT = 'ticket-subject'
    TICKET_UPDATED_AT = 'ticket-updatedat'
    TICKET_CREATED_AT = 'ticket-createdat'
    TICKET_ORIGINAL_ESTIMATION = 'ticket-originalestimation'
    TICKET_FIX_VERSIONS = 'ticket-fixversions'
    TICKET_COMPONENTS = 'ticket-components'
    TICKET_STORYPOINTS = 'ticket-storypoints'


class StringColumnSchema(Schema):
    value: str


class NumberColumnSchema(Schema):
    value: float


class DateColumnSchema(Schema):
    value: date


class SelectColumnSchema(Schema):
    value: list[str]


class CheckboxColumnSchema(Schema):
    value: bool


column_type_schemas = {
    GridColumnTypes.STRING.value: StringColumnSchema,
    GridColumnTypes.NUMBER.value: NumberColumnSchema,
    GridColumnTypes.DATE.value: DateColumnSchema,
    GridColumnTypes.SELECT: SelectColumnSchema,
    GridColumnTypes.STAFF: StringColumnSchema,
    GridColumnTypes.CHECKBOX: CheckboxColumnSchema,
    GridColumnTypes.TICKET: StringColumnSchema,
}


class GridColumnSchema(Schema):
    slug: str
    title: str
    type: GridColumnTypes
    required: bool


class GridRowSchema(Schema):
    id: int
    row: list[CellValueTypes]

    @classmethod
    def serialize(cls, row: dict[str, list[CellValueTypes]]) -> 'GridRowSchema':
        return cls(
            id=row['id'],
            row=row['row'],
        )

    class Config:
        smart_union = True


class GridStructureSchema(Schema):
    sorting: list[str] | None
    fields: list[GridColumnSchema]


class CreateGridSchema(Schema):
    title: str
    page: PageIdentity
    structure: GridStructureSchema | None
    rows: list[list[CellValueTypes]] | None


class PageGridsSchema(Schema):
    id: UUID4
    title: str

    @classmethod
    def serialize(cls, grid) -> 'PageGridsSchema':
        return cls(id=grid.id, title=grid.title)


class GridSchema(Schema):
    id: UUID4
    title: str
    page: PageIdentity
    structure: GridStructureSchema
    rows: list[GridRowSchema]

    @classmethod
    def serialize(cls, grid) -> 'GridSchema':
        return cls(
            id=grid.id,
            page=PageIdentity.serialize(grid.page),
            title=grid.title,
            structure=grid.structure,
            rows=[GridRowSchema.serialize(row) for row in grid.rows],
        )


class GridCellLocation(Schema):
    row_id: int
    column_slug: constr(min_length=1, max_length=255)


class GridCellSchema(Schema):
    location: GridCellLocation
    value: CellValueTypes


class AddRowsSchema(Schema):
    position: int | None
    rows: list[CellValueTypes]


class MoveRowsSchema(Schema):
    position: int
    row_id: int
    rows_count: int | None


class AddColumnsSchema(Schema):
    position: int | None
    columns: list[GridColumnSchema]


class MoveColumnsSchema(Schema):
    position: int
    column_slug: constr(min_length=1, max_length=255)
    columns_count: int | None


class RemoveColumnsSchema(Schema):
    column_slugs: list[constr(min_length=1, max_length=255)]


class RemoveRowsSchema(Schema):
    row_ids: list[int]


class UpdateCellsSchema(Schema):
    cells: list[GridCellSchema]


class UpdateRangeCellsSchema(Schema):
    range_from: GridCellLocation
    range_to: GridCellLocation
    values: list[CellValueTypes]
