from django.shortcuts import get_object_or_404
from pydantic import UUID4

from wiki.acl.check_access import assert_has_access
from wiki.acl.consts import Action
from wiki.api_v2.collections import Collection
from wiki.api_v2.di import di, log_slug
from wiki.api_v2.public.grids.schemas import (
    AddColumnsSchema, AddRowsSchema, CreateGridSchema, GridSchema,
    MoveRowsSchema, MoveColumnsSchema, PageGridsSchema, RemoveColumnsSchema,
    RemoveRowsSchema, UpdateCellsSchema, UpdateRangeCellsSchema,
)
from wiki.api_v2.public.pages.page_identity import resolve_page_identity
from wiki.api_v2.schemas import StatusOk

from wiki.inline_grids.logic import create_grid
from wiki.inline_grids.models import Grid
from wiki.sync.connect.base_organization import BaseOrganization


@di
def get_grid_view(request, organization, idx: UUID4) -> GridSchema:
    grid = get_object_or_404(Grid, pk=idx)
    log_slug(request, slug=grid.page.slug)
    assert_has_access(request.user, organization, grid.page, Action.READ)
    return GridSchema.serialize(grid)


@di
def page_grids_view(request, organization, idx: int) -> Collection[PageGridsSchema]:
    """
    Возвращает список гридов привязанных к странице
    """
    return Collection(results=[], has_next=False, page_id=1)


@di
def create_grid_view(request, organization: BaseOrganization, data: CreateGridSchema) -> GridSchema:
    page = resolve_page_identity(organization, page=data.page)
    log_slug(request, slug=page.slug)
    assert_has_access(request.user, organization, page, Action.READ)

    grid = create_grid(
        user=request.user,
        organization=organization,
        title=data.title,
        page=page,
        structure=data.structure,
        rows=data.rows if data.rows else None,
    )
    return GridSchema.serialize(grid)


@di
def add_rows_view(request, organization: BaseOrganization, idx: UUID4, data: AddRowsSchema):
    grid = get_object_or_404(Grid, pk=idx)
    log_slug(request, slug=grid.page.slug)
    assert_has_access(request.user, organization, grid.page, Action.READ)
    return StatusOk()


@di
def move_rows_view(request, organization: BaseOrganization, idx: UUID4, data: MoveRowsSchema):
    grid = get_object_or_404(Grid, pk=idx)
    log_slug(request, slug=grid.page.slug)
    assert_has_access(request.user, organization, grid.page, Action.READ)
    return StatusOk()


@di
def remove_rows_view(request, organization: BaseOrganization, idx: UUID4, data: RemoveRowsSchema):
    grid = get_object_or_404(Grid, pk=idx)
    log_slug(request, slug=grid.page.slug)
    assert_has_access(request.user, organization, grid.page, Action.READ)
    return StatusOk()


@di
def add_columns_view(request, organization: BaseOrganization, idx: UUID4, data: AddColumnsSchema):
    grid = get_object_or_404(Grid, pk=idx)
    log_slug(request, slug=grid.page.slug)
    assert_has_access(request.user, organization, grid.page, Action.READ)
    return StatusOk()


@di
def move_columns_view(request, organization: BaseOrganization, idx: UUID4, data: MoveColumnsSchema):
    grid = get_object_or_404(Grid, pk=idx)
    log_slug(request, slug=grid.page.slug)
    assert_has_access(request.user, organization, grid.page, Action.READ)
    return StatusOk()


@di
def remove_columns_view(request, organization: BaseOrganization, idx: UUID4, data: RemoveColumnsSchema):
    grid = get_object_or_404(Grid, pk=idx)
    log_slug(request, slug=grid.page.slug)
    assert_has_access(request.user, organization, grid.page, Action.READ)
    return StatusOk()


@di
def update_cells_view(request, organization: BaseOrganization, idx: UUID4, data: UpdateCellsSchema):
    grid = get_object_or_404(Grid, pk=idx)
    log_slug(request, slug=grid.page.slug)
    assert_has_access(request.user, organization, grid.page, Action.READ)
    return StatusOk()


@di
def update_rangecells_view(request, organization: BaseOrganization, idx: UUID4, data: UpdateRangeCellsSchema):
    grid = get_object_or_404(Grid, pk=idx)
    log_slug(request, slug=grid.page.slug)
    assert_has_access(request.user, organization, grid.page, Action.READ)
    return StatusOk()
