from pydantic import Field, constr
from typing import Optional, List

from ninja import Query, Schema
from wiki.acl.check_access import assert_has_access
from wiki.acl.consts import Action
from wiki.api_v2.collections import (
    Collection,
    CollectionFactory,
    OrderDirection,
    PaginationQuery,
)
from wiki.api_v2.di import di, legacy_org_ctx
from wiki.api_v2.public.me.favorites.schemas import (
    AutoBookmarkSchema,
    BookmarkSchema,
    BookmarkPageSchema,
    CreateBookmarkSchema,
    CreateDeleteTagSchema,
    CreateTagResponse,
    TagSchema,
    UpdateBookmarkSchema,
)
from wiki.api_v2.public.pages.page_identity import resolve_page_identity
from wiki.api_v2.schemas import DELETED
from wiki.favorites.consts import AutoBookmarkType
from wiki.favorites.logic import (
    create_bookmark,
    create_tags,
    delete_bookmark,
    delete_tags,
    get_user_autobookmarks,
    get_user_bookmarks,
    update_bookmark,
)
from wiki.favorites.models import AutoBookmark, Bookmark, Tag
from wiki.pages.models import Page
from wiki.pages.utils.resurrect import restore_deleted_slug
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.users.user_data_repository import USER_DATA_REPOSITORY
from wiki.users.consts import UserFeatureCode
from wiki.users.logic.features import assert_feature_is_not_processing


def serialize_bookmark(b: Bookmark) -> BookmarkSchema:
    return BookmarkSchema(
        id=b.id,
        page=serialize_page(b.page),
        created_at=b.created_at,
        tags=[serialize_tag(tag) for tag in b.tags.all().order_by('name')],
    )


def serialize_autobookmark(b: AutoBookmark) -> AutoBookmarkSchema:
    return AutoBookmarkSchema(
        id=b.id,
        page=serialize_page(b.page),
        created_at=b.created_at,
        bookmark_type=b.bookmark_type,
    )


def serialize_tag(t: Tag) -> TagSchema:
    return TagSchema(
        id=t.id,
        name=t.name,
    )


def serialize_page(page: Page) -> BookmarkPageSchema:
    slug = page.slug if page.is_active else restore_deleted_slug(page.slug)
    return BookmarkPageSchema(
        id=page.id,
        is_active=page.is_active,
        title=page.title,
        slug=slug,
        author=USER_DATA_REPOSITORY.orm_to_user_schema(page.owner),
        modified_at=page.modified_at,
        created_at=page.created_at,
    )


ordering_field = {
    'page.title': ['page__title'],
    'page.slug': ['page__supertag'],
    'page.modified_at': ['page__modified_at'],
    'page.created_at': ['page__created_at'],
    'page.author': ['page__owner__staff__last_name', 'page__owner__staff__first_name'],
    'created_at': ['created_at'],
}

CollectionBookmarkSchema = (
    CollectionFactory(name='bookmark')
    .with_ordering(ordering_field)
    .default_ordering(['created_at'], direction=OrderDirection.DESC)
)


@di
def autobookmarks_view(
    request,
    organization: BaseOrganization,
    bookmark_type: Optional[AutoBookmarkType] = None,
    q: Optional[constr(max_length=255)] = Query(None, description='Поиск по заголовку'),
    ordering: CollectionBookmarkSchema.ordering = Query(...),
    pagination: PaginationQuery = Query(...),
) -> Collection[AutoBookmarkSchema]:
    assert_feature_is_not_processing(request.user, UserFeatureCode.DATA_UI_WEB)
    bookmarks = get_user_autobookmarks(request.user, organization, bookmark_type)

    if q:
        bookmarks = bookmarks.filter(page__title__icontains=q)

    return CollectionBookmarkSchema.ordered_build(
        bookmarks, serializer=serialize_autobookmark, pagination=pagination, ordering=ordering
    )


class BookmarkFilters(Schema):
    q: Optional[constr(max_length=255)] = Field(None, description='Поиск по заголовку')
    tags: List[str] = Field(
        None,
        alias='tags[]',  # Важно! иначе будет tags=boo&tags=bar без скобочек
        description='Фильтровать по тегам. Если переданы несколько `tags[]=boo&tags[]=bar` будут '
        'отображены только те закладки на которых есть все теги ',
    )


@di
def bookmarks_view(
    request,
    organization: BaseOrganization,
    ordering: CollectionBookmarkSchema.ordering = Query(...),
    filters: BookmarkFilters = Query(...),
    pagination: PaginationQuery = Query(...),
) -> Collection[BookmarkSchema]:
    assert_feature_is_not_processing(request.user, UserFeatureCode.DATA_UI_WEB)
    bookmarks = get_user_bookmarks(request.user, organization, filters.tags)

    if filters.q:
        bookmarks = bookmarks.filter(page__title__icontains=filters.q)

    return CollectionBookmarkSchema.ordered_build(
        bookmarks, serializer=serialize_bookmark, pagination=pagination, ordering=ordering
    )


@di
@legacy_org_ctx
def create_bookmark_view(request, organization: BaseOrganization, data: CreateBookmarkSchema) -> BookmarkSchema:
    assert_feature_is_not_processing(request.user, UserFeatureCode.DATA_UI_WEB)
    page = resolve_page_identity(organization, page=data.page)
    assert_has_access(request.user, organization, page, Action.READ)

    bookmark = create_bookmark(request.user, organization, page, data.tags)
    return serialize_bookmark(bookmark)


@di
def update_bookmark_view(
    request, organization: BaseOrganization, idx: int, data: UpdateBookmarkSchema
) -> BookmarkSchema:
    assert_feature_is_not_processing(request.user, UserFeatureCode.DATA_UI_WEB)
    bookmark = update_bookmark(idx, request.user, organization, data.tags)
    return serialize_bookmark(bookmark)


@di
def delete_bookmark_view(request, organization: BaseOrganization, idx: int):
    assert_feature_is_not_processing(request.user, UserFeatureCode.DATA_UI_WEB)
    delete_bookmark(idx, request.user, organization)
    return DELETED


@di
def tags_view(
    request,
    organization: BaseOrganization,
    pagination: PaginationQuery = Query(...),
    q: Optional[constr(max_length=255)] = Query(None, description='Фильтровать по имени'),
) -> Collection[TagSchema]:
    assert_feature_is_not_processing(request.user, UserFeatureCode.DATA_UI_WEB)
    tags = Tag.objects.filter(user=request.user, org=organization.as_django_model())
    if q:
        tags = tags.filter(name__startswith=q)
    return (
        CollectionFactory()
        .default_ordering(['name'])
        .ordered_build(
            qs=tags,
            serializer=serialize_tag,
            pagination=pagination,
        )
    )


@di
def create_tags_view(request, organization: BaseOrganization, data: CreateDeleteTagSchema) -> CreateTagResponse:
    assert_feature_is_not_processing(request.user, UserFeatureCode.DATA_UI_WEB)
    tags = create_tags(request.user, organization, data.tags)
    return CreateTagResponse(results=[serialize_tag(tag) for tag in tags])


@di
def delete_tags_view(request, organization: BaseOrganization, data: CreateDeleteTagSchema):
    assert_feature_is_not_processing(request.user, UserFeatureCode.DATA_UI_WEB)
    delete_tags(request.user, organization, data.tags)
    return DELETED
