import waffle

from django.conf import settings

from wiki.api_core.waffle_switches import DEFAULT_NEW_UI
from wiki.api_v2.di import legacy_org_ctx
from wiki.api_v2.public.me.exceptions import UserFeatureChangeForbidden
from wiki.api_v2.public.me.schemas import UpdateUserFeatureSchema, UserFeatureResponse, UserFeatureSchema
from wiki.api_v2.schemas import StatusOk
from wiki.users.consts import UserFeatureCode, UserFeatureStatus
from wiki.users.logic import can_change_ui
from wiki.users.models import UserFeature
from wiki.users.logic.features import enable_feature, disable_feature


def serialize_user_feature(feature_code: UserFeatureCode, status: UserFeatureStatus) -> UserFeatureSchema:
    return UserFeatureSchema(
        code=feature_code,
        status=status,
    )


@legacy_org_ctx
def get_user_features_view(request) -> UserFeatureResponse:
    user_features = UserFeature.objects.filter(user=request.user).order_by('feature_code')
    features = {f.value: UserFeatureStatus.DISABLED for f in UserFeatureCode}

    if waffle.switch_is_active(DEFAULT_NEW_UI):
        enabled = can_change_ui(request.user)
        status = UserFeatureStatus.ENABLED if enabled else UserFeatureStatus.DISABLED
        features[UserFeatureCode.DATA_UI_WEB.value] = status

    for feature in user_features:
        features[feature.feature_code] = feature.status

    return UserFeatureResponse(results=[serialize_user_feature(code, status) for code, status in features.items()])


@legacy_org_ctx
def enable_user_feature_view(request, data: UpdateUserFeatureSchema):
    if data.feature_code == UserFeatureCode.DATA_UI_WEB:
        if request.user.is_robot() or request.user.username in settings.FORBIDDEN_USERS_TO_CHANGE_UI:
            raise UserFeatureChangeForbidden

    enable_feature(request.user, data.feature_code)
    return StatusOk()


@legacy_org_ctx
def disable_user_feature_view(request, data: UpdateUserFeatureSchema):
    disable_feature(request.user, data.feature_code)
    return StatusOk()
