import json
import logging
import waffle
from enum import Enum
from typing import Optional, Dict, Any

from django.conf import settings as django_settings
from ninja import Schema

from wiki.api_core.utils import is_tvm_authentication
from wiki.api_core.waffle_switches import DEFAULT_NEW_UI
from wiki.api_v2.exceptions import TVM2TicketRequired, ValidationError, BadRequest
from wiki.api_v2.schemas import AllOptional
from wiki.pages.access import user_group_ids
from wiki.users.logic import set_user_lang_ui, can_change_ui
from wiki.utils import staff
from wiki.utils.django_redlock.redlock import RedisLock

logger = logging.getLogger(__name__)


class CodeThemeEnum(str, Enum):
    GITHUB = 'github'
    IDEA = 'idea'
    SOLARIZED_LIGHT = 'solarized_light'
    SOLARIZED_DARK = 'solarized_dark'
    DARK = 'dark'


class UiThemeEnum(str, Enum):
    LIGHT = 'light'
    DARK = 'dark'
    SYSTEM = 'system'


class UserSettingsSchema(Schema):
    propose_content_translation: Optional[bool]
    use_nodejs_frontend: Optional[bool]
    use_full_width_content: Optional[bool]
    use_new_wf: Optional[bool]
    language: Optional[str]
    code_theme: Optional[CodeThemeEnum]
    ui_theme: Optional[UiThemeEnum]
    data_ui_web: Optional[bool]
    untyped_settings: Dict[Any, Any]
    show_features_pages: bool


class UserSettingsPatchSchema(UserSettingsSchema, metaclass=AllOptional):
    pass


DEFAULT_SETTINGS = UserSettingsSchema(
    code_theme=CodeThemeEnum.GITHUB,
    propose_content_translation=False,
    use_nodejs_frontend=False,
    use_full_width_content=False,
    use_new_wf=True,
    ui_theme=UiThemeEnum.LIGHT,
    data_ui_web=False,
    untyped_settings={},
    show_features_pages=False,
)

if not django_settings.IS_INTRANET:
    DEFAULT_SETTINGS.language = 'ru'

DEFAULT_SETTINGS_DICT = DEFAULT_SETTINGS.dict()


def get_user_settings(user, request_language_code: str = 'ru') -> UserSettingsSchema:
    profile = getattr(user, 'profile', {})
    settings = {}

    for setting, default in DEFAULT_SETTINGS_DICT.items():
        if setting == 'data_ui_web' and waffle.switch_is_active(DEFAULT_NEW_UI):
            default = can_change_ui(user)

        settings[setting] = profile.get(setting, default)

    # что так как в интранете данные про язык лежат в blackbox,
    # мы не обновляем их в настройках и используем то что пришло из blackbox в миддлвари
    # но это не точно

    if django_settings.IS_INTRANET:
        settings['language'] = request_language_code
        settings['show_features_pages'] = True

        if any(group_id in django_settings.DISABLE_FEATURED_PAGES_GROUP_IDS for group_id in user_group_ids(user.staff)):
            settings['show_features_pages'] = False

    try:
        return UserSettingsSchema.parse_obj(settings)
    except Exception:
        logger.exception(f'Unable to parse user {user} settings; falling back to defaults')
        return DEFAULT_SETTINGS.copy(deep=True)


MAX_SETTINGS_SIZE = 1024 * 4


class UntypedSettingsTooBig(BadRequest):
    error_code = 'UNTYPED_SETTINGS_OVER_LIMIT'
    debug_message = 'Untyped settings is too big, operation aborted'


def update_user_settings(user, data):
    """
    Заменяем поля, но патчим untyped_settings
    """
    for key, value in data.items():
        if key not in DEFAULT_SETTINGS_DICT:
            raise ValidationError()

        if key == 'untyped_settings':
            val = user.profile.get('untyped_settings', {})
            val.update(value)
            if len(json.dumps(val)) > MAX_SETTINGS_SIZE:
                raise UntypedSettingsTooBig()
            value = val

        if key == 'show_features_pages':  # because it is determined by belonging to a group
            continue

        user.profile[key] = value
    user.save()


def get_settings_view(request) -> UserSettingsSchema:
    """
    Настройки в БД хранят только те поля которые отличаются от дефолтных
    Мы таким образом не только экономим место, но и обеспечиваем динамику смены дефолта для всех
    """
    return get_user_settings(request.user, request.LANGUAGE_CODE)


def patch_settings_view(request, data: UserSettingsPatchSchema) -> UserSettingsSchema:
    """
    Обновить настройки, можно передать только те которые изменилась;

    Настройки подновляются, untyped_settings патчится - т.е. то что было
    до этого не пропадет:

    ```
    Было
    {
      'popup_shown': False,
      'agreement_shown': False,
    }
    Patch:
    {
      'agreement_shown': True,
      'foo': 123
    }
    Будет
    {
      'popup_shown': False,
      'agreement_shown': True,
      'foo': 123
    }
    ```

    На untyped_settings есть общее ограничение по объему, при привышении которого будет UNTYPED_SETTINGS_OVER_LIMIT
    """

    with RedisLock(f'untyped_settings_{request.user.id}'):
        return _patch_settings_view(request, data)


def _patch_settings_view(request, data: UserSettingsPatchSchema) -> UserSettingsSchema:
    user = request.user
    request_dict = data.dict(exclude_unset=True)

    # смена языка — это запрос в стафф, так что обрабатываем отдельно
    if django_settings.IS_INTRANET and 'language' in request_dict:
        # для смены языка нам нужен пользовательский тикет
        if not (is_tvm_authentication(request) and request.user_auth.tvm2_user_ticket):
            raise TVM2TicketRequired()

        language = request_dict.pop('language')
        staff.change_user_language(
            user=user,
            language=language,
            user_auth=request.user_auth,
        )

        # сохраняем язык сразу в своей базе, так ждать обновления из стаффа - это долго.
        set_user_lang_ui(user, language)

    if request_dict:
        update_user_settings(user, request_dict)

    return get_user_settings(request.user, request.LANGUAGE_CODE)


def reset_untyped_settings_view(request) -> UserSettingsSchema:
    user = request.user

    with RedisLock(f'untyped_settings_{user.id}'):
        user.profile['untyped_settings'] = {}
        user.save()

    return get_user_settings(request.user, request.LANGUAGE_CODE)
