from typing import Optional
from pydantic import constr

from django.db import transaction
from ninja import Query
from wiki.acl.check_access import assert_has_access
from wiki.acl.consts import Action
from wiki.api_v2.collections import PaginationQuery, Collection, CollectionFactory, OrderDirection
from wiki.api_v2.di import di, legacy_org_ctx
from wiki.api_v2.exceptions import NotFound
from wiki.api_v2.public.me.subscriptions.schemas import (
    CreateSubscriptionSchema,
    DeleteSubscriptionsSchema,
    SubscribedPageSchema,
    SubscriptionSchema,
    SubscriptionType,
)
from wiki.api_v2.public.pages.page_identity import resolve_page_identity, resolve_page_identities, PageIdentity
from wiki.api_v2.schemas import DELETED
from wiki.pages.models import Page
from wiki.pages.utils.resurrect import restore_deleted_slug
from wiki.subscriptions.logic import create_subscription, get_subscriptions_to_cluster_above_batch
from wiki.subscriptions.models import Subscription
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.users.user_data_repository import USER_DATA_REPOSITORY
from wiki.users.consts import UserFeatureCode
from wiki.users.logic.features import assert_feature_is_not_processing


def serialize_page(page: Page) -> SubscribedPageSchema:
    slug = page.slug if page.is_active else restore_deleted_slug(page.slug)

    return SubscribedPageSchema(
        id=page.id,
        title=page.title,
        slug=slug,
        owner=USER_DATA_REPOSITORY.orm_to_user_schema(page.owner),
        modified_at=page.modified_at,
        created_at=page.created_at,
        is_active=page.is_active,
    )


def serialize(subscription: Subscription) -> SubscriptionSchema:
    return SubscriptionSchema(
        id=subscription.id,
        type=subscription.type,
        is_cluster=subscription.is_cluster,
        page=serialize_page(subscription.page),
        created_at=subscription.created_at,
    )


ordering_field = {
    'page.title': ['page__title'],
    'page.slug': ['page__supertag'],
    'page.modified_at': ['page__modified_at'],
    'page.created_at': ['page__created_at'],
    'page.author': ['page__owner__staff__last_name', 'page__owner__staff__first_name'],
    'created_at': ['created_at'],
}

CollectionSubscriptionSchema = (
    CollectionFactory(name='subscription')
    .with_ordering(ordering_field)
    .default_ordering(['created_at'], direction=OrderDirection.DESC)
)


@di
@legacy_org_ctx
def get_subscriptions_view(
    request,
    organization: BaseOrganization,
    type: Optional[SubscriptionType] = None,
    q: Optional[constr(max_length=255)] = Query(None, description='Поиск по заголовку'),
    ordering: CollectionSubscriptionSchema.ordering = Query(...),
    pagination: PaginationQuery = Query(...),
) -> Collection[SubscriptionSchema]:
    """Get user subscriptions filtered by type or not. Ordered by page slug"""

    assert_feature_is_not_processing(request.user, UserFeatureCode.DATA_UI_WEB)
    qs = Subscription.objects.filter(user=request.user, page__org=organization.as_django_model())

    if type:
        qs = qs.filter(type=type)
    if q:
        qs = qs.filter(page__title__icontains=q)

    return CollectionSubscriptionSchema.ordered_build(
        qs, serializer=serialize, pagination=pagination, ordering=ordering
    )


@di
@legacy_org_ctx
def create_subscription_view(
    request,
    organization: BaseOrganization,
    data: CreateSubscriptionSchema,
) -> SubscriptionSchema:
    """
    Create a user subscription to a page. Recreates an existing one.

    And, if the new subscription is to a cluster, deletes all the nested subscriptions
    """

    assert_feature_is_not_processing(request.user, UserFeatureCode.DATA_UI_WEB)
    page = resolve_page_identity(organization, page=data.page)
    assert_has_access(request.user, organization, page, Action.READ)
    subscr = create_subscription(user=request.user, page=page, type_=SubscriptionType.MY, is_cluster=data.is_cluster)
    return serialize(subscr)


@di
@legacy_org_ctx
def delete_subscriptions_view(request, organization: BaseOrganization, data: DeleteSubscriptionsSchema):
    """Delete user subscriptions to pages"""

    assert_feature_is_not_processing(request.user, UserFeatureCode.DATA_UI_WEB)
    pages = resolve_page_identities(organization, actual_pages=data.pages, only_active=False)
    with transaction.atomic():
        cluster_roots = get_subscriptions_to_cluster_above_batch(request.user, pages)
        subscriptions = Subscription.objects.filter(user=request.user, page__in=pages)

        if subscriptions.count() != len(pages) and data.strict:
            exisiting_pages = {s.page for s in subscriptions}
            raise NotFound(
                'Some subscriptions not found',
                details={'missing': [PageIdentity.from_orm(p) for p in (set(pages) - exisiting_pages)]},
            )

        subscriptions.delete()
        for cluster_root in cluster_roots:
            cluster_root.delete()

    return DELETED
