from itertools import chain

from django.conf import settings
from django.db import connection
from ninja import Query

from wiki.api_v2.collections import Collection, CollectionFactory, OrderDirection, PaginationQuery
from wiki.api_v2.di import legacy_org_ctx, di, log_slug
from wiki.api_v2.public.navigation_tree.exceptions import NavTreeBadRequest
from wiki.api_v2.public.navigation_tree.schemas import NavTreeNodeSchema, NodeType, NavTreeSchema
from wiki.api_v2.public.pages.page.getters import get_cluster_schema_from_slug
from wiki.api_v2.schemas import Slug, SlugOrEmpty
from wiki.pages.access import ACCESS_COMMON, ACCESS_RESTRICTED, ACCESS_UNLIMITED
from wiki.pages.access.access_status import get_bulk_access_status_of_pages
from wiki.pages.models import Page
from wiki.pages.models.consts import DB_PAGE_TYPES_MAP, PageType
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.sync.connect.org_ctx import org_ctx
from wiki.utils.db import namedtuple_fetchall

CollectionNavtreeSchema = CollectionFactory(name='navtree').default_ordering(['rank'], OrderDirection.DESC)

COW_SLUGS = settings.COPY_ON_WRITE_TAGS
FIXED_SLUGS = settings.NAVIGATION_TREE_FIXED_SLUGS
VALID_ACCESS = {ACCESS_COMMON, ACCESS_RESTRICTED, ACCESS_UNLIMITED}


def _serialize_page(page: Page, user_has_access: bool = True, has_children: bool = False, is_fixed: bool = False):
    if user_has_access:
        return NavTreeNodeSchema(
            id=page.id,
            slug=page.slug,
            title=page.title,
            type=NodeType.PAGE,
            has_children=has_children,
            page_type=DB_PAGE_TYPES_MAP.get(page.page_type),
            is_fixed=is_fixed,
        )
    else:
        return NavTreeNodeSchema(
            id=page.id,
            slug=page.slug,
            title='***',  # page.slug.split('/')[-1],
            type=NodeType.LOCKED,
            has_children=False,
            page_type=DB_PAGE_TYPES_MAP.get(page.page_type),
            is_fixed=is_fixed,
        )


def cow_node_by_slug(organization, slug, lang):
    if lang not in COW_SLUGS[slug]:
        lang = 'en'

    children_count = (
        organization.get_active_pages().filter(supertag__startswith=f'{slug}/', redirects_to_id=None).count()
    )
    is_fixed = slug in FIXED_SLUGS

    return NavTreeNodeSchema(
        id=COW_SLUGS[slug]['id'],
        slug=slug,
        title=COW_SLUGS[slug][lang]['title'],
        type=NodeType.COW,
        has_children=bool(children_count),
        page_type=PageType.REGULAR_PAGE,
        is_fixed=is_fixed,
    )


def fetch_cow_nodes(organization, page_depth, lang):
    cow_nodes = []
    pages_lut = set(organization.get_active_pages().filter(supertag__in=COW_SLUGS).values_list('supertag', flat=True))
    for slug in COW_SLUGS:
        if not page_depth == slug.count('/') + 1:
            continue

        if slug in pages_lut:
            continue

        # На том же уровне дерева есть copy_on_write страница
        # и она еще не создана (редиректы считаются существующими страницами)
        cow_nodes.append(cow_node_by_slug(organization, slug, lang))

    return cow_nodes


def is_first_level(slug):
    return slug == ''


def _order_fixed_nodes(nodes: list[NavTreeNodeSchema], fixed_slugs: list[str]) -> list[NavTreeNodeSchema]:
    indexes = {v: i for i, v in enumerate(fixed_slugs)}
    last_index = len(fixed_slugs)
    return sorted(nodes, key=lambda node: indexes.get(node.slug, last_index))


def __build_org_filter_clause(org, sql_params):
    if org is not None:
        sql_params['org_id'] = org.id
        return 'org_id = %(org_id)s'
    return 'org_id IS NULL'


def __build_supertag_filter_clause(parent, sql_params):
    if is_first_level(parent):
        sql_params |= {
            'siblings_prefix': '',
        }
        return ''

    sql_params |= {
        'siblings_prefix': f'{parent}/',
        'siblings_like': f'{parent}/%',
    }
    return 'AND supertag LIKE %(siblings_like)s'


def __build_breadcrumb_branch_clause(parent_slug, breadcrumbs_branch_slug, sql_params):
    assert (
        is_first_level(parent_slug)
        or breadcrumbs_branch_slug.startswith(parent_slug + '/')
        or breadcrumbs_branch_slug == parent_slug
    ), 'breadcrumbs_branch_slug must be an ancestor of parent_slug'
    sql_params['highlighed_slug_like'] = breadcrumbs_branch_slug + '/%'

    cluster = get_cluster_schema_from_slug(breadcrumbs_branch_slug)

    cluster_prefix = ''
    breadcrumb_branch_slugs_sorted = []
    if cluster.slug != '':
        cluster_prefix = cluster.slug + '/'
        breadcrumb_branch_slugs_sorted.append(cluster.slug)

    sections = [section for section in breadcrumbs_branch_slug[len(cluster_prefix) :].split('/') if not section == '']
    for i in range(0, len(sections)):
        breadcrumb_branch_slugs_sorted.append(cluster_prefix + '/'.join(sections[: i + 1]))

    sql_params['highlighed_slug_parent_chain'] = tuple(breadcrumb_branch_slugs_sorted)

    return breadcrumb_branch_slugs_sorted


def _load_next(
    user,
    organization: BaseOrganization,
    parent_slug: SlugOrEmpty,
    pagination: PaginationQuery = Query(...),
    lang: str = 'en',
) -> Collection[NavTreeNodeSchema]:
    org = organization.as_django_model()
    page_depth = 1 if is_first_level(parent_slug) else parent_slug.count('/') + 2
    parent = parent_slug

    sql_params = {
        'page_depth': page_depth,
    }

    org_filter = __build_org_filter_clause(org, sql_params)
    supertag_filter = __build_supertag_filter_clause(parent, sql_params)

    sql_query = f"""
        SELECT %(siblings_prefix)s || SPLIT_PART(supertag, '/', %(page_depth)s) as slug, COUNT(1) > 1 as has_children
            FROM pages_page
            WHERE status > 0
            AND redirects_to_id IS NULL
            AND {org_filter}
            {supertag_filter}
            GROUP BY slug;"""
    with connection.cursor() as cursor:
        cursor.execute(sql_query, sql_params)
        rows = namedtuple_fetchall(cursor)

    children_lut = {row.slug: row.has_children for row in rows}

    def _serialize(pages, is_fixed=False):
        with org_ctx(organization.as_django_model()):
            access_statuses = get_bulk_access_status_of_pages(
                pages,
                user=user,
            )

        return [
            _serialize_page(
                page, access_statuses.get(page.slug) in VALID_ACCESS, children_lut.get(page.supertag, False), is_fixed
            )
            for page in pages
            if access_statuses.get(page.slug) in VALID_ACCESS
        ]

    qs = organization.get_active_pages().filter(redirects_to=None, depth=page_depth)
    if not is_first_level(parent):
        qs = qs.filter(supertag__startswith=parent + '/')
    if FIXED_SLUGS:
        qs = qs.exclude(supertag__in=FIXED_SLUGS)

    next_chunk = CollectionNavtreeSchema.ordered_build(qs, batch_serializer=_serialize, pagination=pagination)
    if next_chunk.prev_cursor is None:
        # Выше больше ничего нет, нужно добавить fixed и copy_on_write ноды (находящиеся на том же уровне)
        fixed_pages = (
            organization.get_active_pages()
            .filter(supertag__in=FIXED_SLUGS, depth=page_depth)  # allow redirects for fixed
            .order_by('-id')
        )

        fixed_nodes = _serialize(fixed_pages, True)
        cow_nodes = fetch_cow_nodes(organization, page_depth, lang)

        nodes = _order_fixed_nodes(fixed_nodes + cow_nodes, fixed_slugs=FIXED_SLUGS)
        next_chunk.results = nodes + next_chunk.results

    return next_chunk


def _build_navview(
    user,
    organization: BaseOrganization,
    parent_slug: SlugOrEmpty,
    open_around_node_slug: str = None,
    breadcrumbs_branch_slug: str = None,
    page_size: int = 25,
    lang: str = 'en',
) -> NavTreeSchema:
    if (
        open_around_node_slug
        and not is_first_level(parent_slug)
        and not open_around_node_slug.startswith(parent_slug + '/')
    ):
        raise NavTreeBadRequest(debug_message='open_around_node_slug must be a direct child of parent_slug')

    page_depth = 1 if is_first_level(parent_slug) else parent_slug.count('/') + 2

    if breadcrumbs_branch_slug and open_around_node_slug is None:
        _b_parts = breadcrumbs_branch_slug.split('/')
        open_around_node_slug = '/'.join(_b_parts[:page_depth])

    if breadcrumbs_branch_slug and not (
        open_around_node_slug == breadcrumbs_branch_slug
        or breadcrumbs_branch_slug.startswith(open_around_node_slug + '/')
    ):
        raise NavTreeBadRequest(debug_message='breadcrumbs_branch_slug must be an ancestor of parent_slug')

    org = organization.as_django_model()
    parent = parent_slug

    sql_params = {
        'size': page_size + 1,  # to deduce if we have prev/next
    }

    if open_around_node_slug:
        sql_params['page_slug'] = open_around_node_slug

    sql_params['page_depth'] = page_depth

    org_filter = __build_org_filter_clause(org, sql_params)
    supertag_filter = __build_supertag_filter_clause(parent, sql_params)

    siblings_subquery = f"""siblings AS (
     SELECT id,
            supertag,
            page_type,
            row_number() over (ORDER BY rank DESC, id DESC)
     FROM pages_page
     WHERE status > 0
       AND depth = %(page_depth)s
       AND redirects_to_id IS NULL
       AND {org_filter}
       {supertag_filter}
     ORDER BY rank DESC, id DESC
    )"""

    if FIXED_SLUGS:
        sql_params['fixed_slugs'] = tuple(FIXED_SLUGS)

        siblings_subquery = f"""siblings AS (
        SELECT id,
                supertag,
                page_type,
                row_number() over (ORDER BY rank DESC, id DESC)
        FROM (SELECT id,
                     supertag,
                     page_type,
                     CASE WHEN supertag IN %(fixed_slugs)s THEN '9' ELSE rank END AS rank
                FROM pages_page
                WHERE status > 0
                AND depth = %(page_depth)s
                -- AND redirects_to_id IS NULL
                AND {org_filter}
                {supertag_filter}
                ORDER BY rank DESC, id DESC
            ) AS p
        )"""

    children_subquery = f"""children AS (
        SELECT %(siblings_prefix)s || SPLIT_PART(supertag, '/', %(page_depth)s) as slug, COUNT(1)>1 as has_children
            FROM pages_page
            WHERE status > 0
            AND redirects_to_id IS NULL
            AND {org_filter}
            {supertag_filter}
            GROUP BY slug
    )"""
    match_subquery = """match AS (
         SELECT row_number
         FROM siblings
         WHERE supertag = %(page_slug)s LIMIT 1
    )"""

    sql_open_first_childrens = """
    (
        SELECT siblings.id as id, siblings.supertag as slug, children.has_children as has_children,
        FALSE as is_highlighted, siblings.page_type as page_type
        FROM siblings JOIN children ON siblings.supertag = children.slug ORDER BY siblings.row_number LIMIT %(size)s
    )
    """
    sql_open_around_node = """
    (
        SELECT siblings.id as id, siblings.supertag as slug, children.has_children as has_children,
        FALSE as is_highlighted, siblings.page_type as page_type
        FROM siblings JOIN children ON siblings.supertag = children.slug
        WHERE ABS(COALESCE((SELECT * from match), 0)  - siblings.row_number) <= %(size)s
        ORDER BY siblings.row_number
    )
    """

    breadcrumb_branch_sql = f"""
    UNION ALL
    (
        SELECT page.id as id, page.supertag as slug,
        EXISTS(SELECT 1
            FROM pages_page
            WHERE supertag LIKE %(highlighed_slug_like)s
            AND status > 0
            AND redirects_to_id IS NULL
            AND {org_filter}) as has_children,
        TRUE as is_highlighted,
        page.page_type as page_type
        FROM pages_page as page
        WHERE supertag IN %(highlighed_slug_parent_chain)s
        AND status > 0
        AND redirects_to_id IS NULL
        AND {org_filter}
        ORDER BY depth
    )
    """

    subqueries = [siblings_subquery, children_subquery]

    if open_around_node_slug:
        subqueries.append(match_subquery)
        select = sql_open_around_node
    else:
        select = sql_open_first_childrens

    if breadcrumbs_branch_slug is not None:
        breadcrumb_branch_slugs_sorted = __build_breadcrumb_branch_clause(
            parent_slug, breadcrumbs_branch_slug, sql_params
        )
        union_all = breadcrumb_branch_sql
        breadcrumbs_branch_nodes = []
    else:
        breadcrumb_branch_slugs_sorted = []
        breadcrumbs_branch_nodes = None
        union_all = ''

    subqueries = ',\n'.join(subqueries)

    sql_query = f"""
        WITH {subqueries}
        {select}
        {union_all};
    """

    page_offset = None

    with connection.cursor() as cursor:
        cursor.execute(sql_query, sql_params)
        all_rows = namedtuple_fetchall(cursor)

    rows = [row for row in all_rows if not row.is_highlighted]
    children_lut = {row.slug: row.has_children for row in all_rows if not row.is_highlighted}

    breadcrumbs_branch = [row for row in all_rows if row.is_highlighted]
    if len(breadcrumbs_branch) > 0:
        children_lut[breadcrumbs_branch[-1].slug] = breadcrumbs_branch[-1].has_children
        for row in breadcrumbs_branch[:-1]:
            children_lut[row.slug] = True

    if open_around_node_slug:
        for idx, row in enumerate(rows):
            if row.slug == open_around_node_slug:
                page_offset = idx

    if page_offset is None:
        cut_rows = rows[:page_size]
        has_prev = False
        has_next = len(rows) == (page_size + 1)
    else:
        has_prev = page_offset == (page_size + 1)
        has_next = (len(rows) - page_offset - 1) == (page_size + 1)
        cut_rows = rows[max(0, page_offset - page_size) : min(page_offset + page_size + 1, len(rows))]

    # get pages + filter cut_rows by access
    ids = [row.id for row in chain(cut_rows, breadcrumbs_branch)]
    nodes = []
    readable_pages = []

    with org_ctx(org):
        pages = list(
            organization.get_active_pages()
            .filter(id__in=ids)
            .prefetch_related('authors')
            .only('id', 'supertag', 'tag', 'authors', 'rank', 'title')
        )
        page_lut = {page.slug: page for page in pages}
        access_lut = get_bulk_access_status_of_pages(pages, user=user)

        def _maybe_serialize_page(slug, page_lut, access_lut, children_lut, is_fixed):
            if slug in page_lut:
                return _serialize_page(
                    page_lut[slug], access_lut.get(slug) in VALID_ACCESS, children_lut.get(slug), is_fixed
                )
            elif slug in COW_SLUGS:
                return cow_node_by_slug(organization, slug, lang)
            return NavTreeNodeSchema(
                id=None, slug=slug, title=slug.split('/')[-1], type=NodeType.GAP, has_children=False, is_fixed=is_fixed
            )

        for row in cut_rows:
            if access_lut.get(row.slug) not in VALID_ACCESS:
                continue

            is_fixed = row.slug in FIXED_SLUGS
            node = _maybe_serialize_page(row.slug, page_lut, access_lut, children_lut, is_fixed)
            if row.slug in page_lut:
                page = page_lut[row.slug]
                if row.slug not in FIXED_SLUGS:
                    readable_pages.append(page)

                nodes.append(node)

        if not has_prev:
            # copy on write страницы всегда вверху
            cow_nodes = fetch_cow_nodes(organization, page_depth, lang)
            nodes = cow_nodes + nodes

        for subslug in breadcrumb_branch_slugs_sorted:
            subnode = _maybe_serialize_page(subslug, page_lut, access_lut, children_lut, False)
            breadcrumbs_branch_nodes.append(subnode)

    collection = Collection[NavTreeNodeSchema](
        results=_order_fixed_nodes(nodes, fixed_slugs=FIXED_SLUGS),
        has_next=has_next,
        page_id=1,
        next_cursor=CollectionNavtreeSchema.create_next_cursor(readable_pages) if has_next else None,
        prev_cursor=CollectionNavtreeSchema.create_prev_cursor(readable_pages) if has_prev else None,
    )

    return NavTreeSchema(children=collection, breadcrumbs_branch=breadcrumbs_branch_nodes)


@di
@legacy_org_ctx
def navtree_view(
    request,
    organization: BaseOrganization,
    parent_slug: SlugOrEmpty,
    open_around_node_slug: Slug = None,
    breadcrumbs_branch_slug: Slug = None,
    page_size: int = 50,
) -> NavTreeSchema:
    """
    Первоначальная подгрузка навигационного дерева.
    - Поймем к какому кластеру принадлежит слаг,
    - раскроем узлы в окрестности предка который является первым потомком в кластере
    - подгрузим и подсветим путь к узлу
    """
    log_slug(request, slug=parent_slug)

    return _build_navview(
        request.user,
        organization,
        parent_slug=parent_slug,
        open_around_node_slug=open_around_node_slug,
        breadcrumbs_branch_slug=breadcrumbs_branch_slug,
        page_size=page_size,
        lang=request.LANGUAGE_CODE,
    )


@di
@legacy_org_ctx
def load_more_view(
    request, organization: BaseOrganization, parent_slug: SlugOrEmpty, pagination: PaginationQuery = Query(...)
) -> Collection[NavTreeNodeSchema]:
    log_slug(request, slug=parent_slug)
    return _load_next(request.user, organization, parent_slug, pagination, lang=request.LANGUAGE_CODE)
